﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kafka/KafkaRequest.h>
#include <aws/kafka/Kafka_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Kafka {
namespace Model {

/**
 */
class ListClustersRequest : public KafkaRequest {
 public:
  AWS_KAFKA_API ListClustersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListClusters"; }

  AWS_KAFKA_API Aws::String SerializePayload() const override;

  AWS_KAFKA_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   *
          <p>Specify a prefix of the name of the clusters that you want to
   * list. The service lists all the clusters whose names start with this
   * prefix.</p>

   */
  inline const Aws::String& GetClusterNameFilter() const { return m_clusterNameFilter; }
  inline bool ClusterNameFilterHasBeenSet() const { return m_clusterNameFilterHasBeenSet; }
  template <typename ClusterNameFilterT = Aws::String>
  void SetClusterNameFilter(ClusterNameFilterT&& value) {
    m_clusterNameFilterHasBeenSet = true;
    m_clusterNameFilter = std::forward<ClusterNameFilterT>(value);
  }
  template <typename ClusterNameFilterT = Aws::String>
  ListClustersRequest& WithClusterNameFilter(ClusterNameFilterT&& value) {
    SetClusterNameFilter(std::forward<ClusterNameFilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   *
          <p>The maximum number of results to return in the response. If
   * there are more results, the response includes a NextToken parameter.</p>

   *
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListClustersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   *
          <p>The paginated results marker. When the result of the operation
   * is truncated, the call returns NextToken in the response.
          To get
   * the next batch, provide this token in your next request.</p>

   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListClustersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterNameFilter;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_clusterNameFilterHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Kafka
}  // namespace Aws
