﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/memorydb/MemoryDBRequest.h>
#include <aws/memorydb/MemoryDB_EXPORTS.h>
#include <aws/memorydb/model/Tag.h>

#include <utility>

namespace Aws {
namespace MemoryDB {
namespace Model {

/**
 */
class PurchaseReservedNodesOfferingRequest : public MemoryDBRequest {
 public:
  AWS_MEMORYDB_API PurchaseReservedNodesOfferingRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PurchaseReservedNodesOffering"; }

  AWS_MEMORYDB_API Aws::String SerializePayload() const override;

  AWS_MEMORYDB_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the reserved node offering to purchase.</p>
   */
  inline const Aws::String& GetReservedNodesOfferingId() const { return m_reservedNodesOfferingId; }
  inline bool ReservedNodesOfferingIdHasBeenSet() const { return m_reservedNodesOfferingIdHasBeenSet; }
  template <typename ReservedNodesOfferingIdT = Aws::String>
  void SetReservedNodesOfferingId(ReservedNodesOfferingIdT&& value) {
    m_reservedNodesOfferingIdHasBeenSet = true;
    m_reservedNodesOfferingId = std::forward<ReservedNodesOfferingIdT>(value);
  }
  template <typename ReservedNodesOfferingIdT = Aws::String>
  PurchaseReservedNodesOfferingRequest& WithReservedNodesOfferingId(ReservedNodesOfferingIdT&& value) {
    SetReservedNodesOfferingId(std::forward<ReservedNodesOfferingIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A customer-specified identifier to track this reservation.</p>
   */
  inline const Aws::String& GetReservationId() const { return m_reservationId; }
  inline bool ReservationIdHasBeenSet() const { return m_reservationIdHasBeenSet; }
  template <typename ReservationIdT = Aws::String>
  void SetReservationId(ReservationIdT&& value) {
    m_reservationIdHasBeenSet = true;
    m_reservationId = std::forward<ReservationIdT>(value);
  }
  template <typename ReservationIdT = Aws::String>
  PurchaseReservedNodesOfferingRequest& WithReservationId(ReservationIdT&& value) {
    SetReservationId(std::forward<ReservationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of node instances to reserve.</p>
   */
  inline int GetNodeCount() const { return m_nodeCount; }
  inline bool NodeCountHasBeenSet() const { return m_nodeCountHasBeenSet; }
  inline void SetNodeCount(int value) {
    m_nodeCountHasBeenSet = true;
    m_nodeCount = value;
  }
  inline PurchaseReservedNodesOfferingRequest& WithNodeCount(int value) {
    SetNodeCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to be added to this resource. A tag is a key-value pair. A tag
   * key must be accompanied by a tag value, although null is accepted.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  PurchaseReservedNodesOfferingRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  PurchaseReservedNodesOfferingRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_reservedNodesOfferingId;

  Aws::String m_reservationId;

  int m_nodeCount{0};

  Aws::Vector<Tag> m_tags;
  bool m_reservedNodesOfferingIdHasBeenSet = false;
  bool m_reservationIdHasBeenSet = false;
  bool m_nodeCountHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MemoryDB
}  // namespace Aws
