/*
 * Copyright 2013 The Error Prone Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.errorprone;

import com.google.common.base.CharMatcher;

/**
 * Validates an {@code @BugPattern} annotation for wellformedness.
 *
 * @author eaftan@google.com (Eddie Aftandilian)
 */
public class BugPatternValidator {

  public static void validate(BugPattern pattern) throws ValidationException {
    if (pattern == null) {
      throw new ValidationException("No @BugPattern provided");
    }

    // name must not contain spaces
    if (CharMatcher.whitespace().matchesAnyOf(pattern.name())) {
      throw new ValidationException("Name must not contain whitespace: " + pattern.name());
    }

    // linkType must be consistent with link element.
    switch (pattern.linkType()) {
      case CUSTOM:
        if (pattern.link().isEmpty()) {
          throw new ValidationException("Expected a custom link but none was provided");
        }
        break;
      case AUTOGENERATED:
      case NONE:
        if (!pattern.link().isEmpty()) {
          throw new ValidationException("Expected no custom link but found: " + pattern.link());
        }
        break;
    }
  }
}
