/**
 * Copyright Copyright 2010-17 Simon Andrews
 *
 *    This file is part of FastQC.
 *
 *    FastQC is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    FastQC is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with FastQC; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package uk.ac.babraham.FastQC.Sequence;

import java.io.File;
import java.io.IOException;

import uk.ac.babraham.FastQC.FastQCConfig;
import uk.ac.babraham.FastQC.Utilities.CasavaBasename;
import uk.ac.babraham.FastQC.Utilities.NameFormatException;

public class SequenceFactory {
	/**
	 * 
	 * This option is used when multiple files are to be treated as a group to produce
	 * a single output.  This is currently used for groups of files generated by casava
	 * 
	 * @param files
	 * @return
	 * @throws SequenceFormatException
	 * @throws IOException
	 */
	public static SequenceFile getSequenceFile (File [] files) throws SequenceFormatException, IOException {
		
		/*
		 * We used to build a set of SequenceFile objects to make a sequence group, but we found that
		 * doing this caused too many files to be open simultaneously and caused crashes on large
		 * runs.  We now just pass the files to the group and let it open them as and when they're needed.
		 */
		
		if (files.length == 1) {
			
			if (FastQCConfig.getInstance().casava) {
				try {
					// We do this simply to find out if the casava basename is valid.  If it is then the 
					// Sequencefilegroup is created at the end of this sub.  If it's not then we do a 
					// simple creation without modifying the file name.
					CasavaBasename.getCasavaBasename(files[0].getName());
				}
				catch (NameFormatException nfe) {
					return getSequenceFile(files[0]);
				}
			}
			else {
				return getSequenceFile(files[0]);	
			}
		}
		
		
		return new SequenceFileGroup(files);
		
	}
	
	public static SequenceFile getSequenceFile(File file) throws SequenceFormatException, IOException {
		
		FastQCConfig config = FastQCConfig.getInstance();

//		System.err.println("Format is "+config.sequence_format);
		
		if (config.sequence_format != null) {
			// We're not autodetecting the format, but taking whatever they said
			
			if (config.sequence_format.equals("bam") || config.sequence_format.equals("sam")) {
				return new BAMFile(file,false);				
			}
			else if (config.sequence_format.equals("bam_mapped") || config.sequence_format.equals("sam_mapped")) {
				return new BAMFile(file,true);				
			}
			else if (config.sequence_format.equals("fastq")) {
				return new FastQFile(config,file);
			}
			else {
				throw new SequenceFormatException("Didn't understand format name '"+config.sequence_format+"'");
			}
			
		}
		
		
		// Otherwise we just use the extension on the end of the file name to try to determine
		// the type
		if (file.getName().toLowerCase().endsWith(".bam") || file.getName().toLowerCase().endsWith(".sam") || file.getName().toLowerCase().endsWith(".ubam")) {
			// We default to using all reads
			return new BAMFile(file,false);
		}
		else if (file.getName().toLowerCase().endsWith(".fast5")) {
			return new Fast5File(file);
		}
		else {
			return new FastQFile(config,file);
		}

		
	}
	
	
}
