# -*- coding: utf-8 -*-
from setuptools import setup

setup(
    name='black',
    version='%%PORTVERSION%%',
    description='The uncompromising code formatter.',
    long_description='[![Black Logo](https://raw.githubusercontent.com/psf/black/main/docs/_static/logo2-readme.png)](https://black.readthedocs.io/en/stable/)\n\n<h2 align="center">The Uncompromising Code Formatter</h2>\n\n<p align="center">\n<a href="https://github.com/psf/black/actions"><img alt="Actions Status" src="https://github.com/psf/black/workflows/Test/badge.svg"></a>\n<a href="https://black.readthedocs.io/en/stable/?badge=stable"><img alt="Documentation Status" src="https://readthedocs.org/projects/black/badge/?version=stable"></a>\n<a href="https://coveralls.io/github/psf/black?branch=main"><img alt="Coverage Status" src="https://coveralls.io/repos/github/psf/black/badge.svg?branch=main"></a>\n<a href="https://github.com/psf/black/blob/main/LICENSE"><img alt="License: MIT" src="https://black.readthedocs.io/en/stable/_static/license.svg"></a>\n<a href="https://pypi.org/project/black/"><img alt="PyPI" src="https://img.shields.io/pypi/v/black"></a>\n<a href="https://pepy.tech/project/black"><img alt="Downloads" src="https://pepy.tech/badge/black"></a>\n<a href="https://anaconda.org/conda-forge/black/"><img alt="conda-forge" src="https://img.shields.io/conda/dn/conda-forge/black.svg?label=conda-forge"></a>\n<a href="https://github.com/psf/black"><img alt="Code style: black" src="https://img.shields.io/badge/code%20style-black-000000.svg"></a>\n</p>\n\n> “Any color you like.”\n\n_Black_ is the uncompromising Python code formatter. By using it, you agree to cede\ncontrol over minutiae of hand-formatting. In return, _Black_ gives you speed,\ndeterminism, and freedom from `pycodestyle` nagging about formatting. You will save time\nand mental energy for more important matters.\n\nBlackened code looks the same regardless of the project you\'re reading. Formatting\nbecomes transparent after a while and you can focus on the content instead.\n\n_Black_ makes code review faster by producing the smallest diffs possible.\n\nTry it out now using the [Black Playground](https://black.vercel.app). Watch the\n[PyCon 2019 talk](https://youtu.be/esZLCuWs_2Y) to learn more.\n\n---\n\n**[Read the documentation on ReadTheDocs!](https://black.readthedocs.io/en/stable)**\n\n---\n\n## Installation and usage\n\n### Installation\n\n_Black_ can be installed by running `pip install black`. It requires Python 3.7+ to run.\nIf you want to format Jupyter Notebooks, install with `pip install \'black[jupyter]\'`.\n\nIf you can\'t wait for the latest _hotness_ and want to install from GitHub, use:\n\n`pip install git+https://github.com/psf/black`\n\n### Usage\n\nTo get started right away with sensible defaults:\n\n```sh\nblack {source_file_or_directory}\n```\n\nYou can run _Black_ as a package if running it as a script doesn\'t work:\n\n```sh\npython -m black {source_file_or_directory}\n```\n\nFurther information can be found in our docs:\n\n- [Usage and Configuration](https://black.readthedocs.io/en/stable/usage_and_configuration/index.html)\n\n_Black_ is already [successfully used](https://github.com/psf/black#used-by) by many\nprojects, small and big. _Black_ has a comprehensive test suite, with efficient parallel\ntests, and our own auto formatting and parallel Continuous Integration runner. Now that\nwe have become stable, you should not expect large formatting to changes in the future.\nStylistic changes will mostly be responses to bug reports and support for new Python\nsyntax. For more information please refer to the\n[The Black Code Style](https://black.readthedocs.io/en/stable/the_black_code_style/index.html).\n\nAlso, as a safety measure which slows down processing, _Black_ will check that the\nreformatted code still produces a valid AST that is effectively equivalent to the\noriginal (see the\n[Pragmatism](https://black.readthedocs.io/en/stable/the_black_code_style/current_style.html#ast-before-and-after-formatting)\nsection for details). If you\'re feeling confident, use `--fast`.\n\n## The _Black_ code style\n\n_Black_ is a PEP 8 compliant opinionated formatter. _Black_ reformats entire files in\nplace. Style configuration options are deliberately limited and rarely added. It doesn\'t\ntake previous formatting into account (see\n[Pragmatism](https://black.readthedocs.io/en/stable/the_black_code_style/current_style.html#pragmatism)\nfor exceptions).\n\nOur documentation covers the current _Black_ code style, but planned changes to it are\nalso documented. They\'re both worth taking a look:\n\n- [The _Black_ Code Style: Current style](https://black.readthedocs.io/en/stable/the_black_code_style/current_style.html)\n- [The _Black_ Code Style: Future style](https://black.readthedocs.io/en/stable/the_black_code_style/future_style.html)\n\nChanges to the _Black_ code style are bound by the Stability Policy:\n\n- [The _Black_ Code Style: Stability Policy](https://black.readthedocs.io/en/stable/the_black_code_style/index.html#stability-policy)\n\nPlease refer to this document before submitting an issue. What seems like a bug might be\nintended behaviour.\n\n### Pragmatism\n\nEarly versions of _Black_ used to be absolutist in some respects. They took after its\ninitial author. This was fine at the time as it made the implementation simpler and\nthere were not many users anyway. Not many edge cases were reported. As a mature tool,\n_Black_ does make some exceptions to rules it otherwise holds.\n\n- [The _Black_ code style: Pragmatism](https://black.readthedocs.io/en/stable/the_black_code_style/current_style.html#pragmatism)\n\nPlease refer to this document before submitting an issue just like with the document\nabove. What seems like a bug might be intended behaviour.\n\n## Configuration\n\n_Black_ is able to read project-specific default values for its command line options\nfrom a `pyproject.toml` file. This is especially useful for specifying custom\n`--include` and `--exclude`/`--force-exclude`/`--extend-exclude` patterns for your\nproject.\n\nYou can find more details in our documentation:\n\n- [The basics: Configuration via a file](https://black.readthedocs.io/en/stable/usage_and_configuration/the_basics.html#configuration-via-a-file)\n\nAnd if you\'re looking for more general configuration documentation:\n\n- [Usage and Configuration](https://black.readthedocs.io/en/stable/usage_and_configuration/index.html)\n\n**Pro-tip**: If you\'re asking yourself "Do I need to configure anything?" the answer is\n"No". _Black_ is all about sensible defaults. Applying those defaults will have your\ncode in compliance with many other _Black_ formatted projects.\n\n## Used by\n\nThe following notable open-source projects trust _Black_ with enforcing a consistent\ncode style: pytest, tox, Pyramid, Django, Django Channels, Hypothesis, attrs,\nSQLAlchemy, Poetry, PyPA applications (Warehouse, Bandersnatch, Pipenv, virtualenv),\npandas, Pillow, Twisted, LocalStack, every Datadog Agent Integration, Home Assistant,\nZulip, Kedro, OpenOA, FLORIS, ORBIT, WOMBAT, and many more.\n\nThe following organizations use _Black_: Facebook, Dropbox, KeepTruckin, Mozilla, Quora,\nDuolingo, QuantumBlack, Tesla.\n\nAre we missing anyone? Let us know.\n\n## Testimonials\n\n**Mike Bayer**, [author of `SQLAlchemy`](https://www.sqlalchemy.org/):\n\n> I can\'t think of any single tool in my entire programming career that has given me a\n> bigger productivity increase by its introduction. I can now do refactorings in about\n> 1% of the keystrokes that it would have taken me previously when we had no way for\n> code to format itself.\n\n**Dusty Phillips**,\n[writer](https://smile.amazon.com/s/ref=nb_sb_noss?url=search-alias%3Daps&field-keywords=dusty+phillips):\n\n> _Black_ is opinionated so you don\'t have to be.\n\n**Hynek Schlawack**, [creator of `attrs`](https://www.attrs.org/), core developer of\nTwisted and CPython:\n\n> An auto-formatter that doesn\'t suck is all I want for Xmas!\n\n**Carl Meyer**, [Django](https://www.djangoproject.com/) core developer:\n\n> At least the name is good.\n\n**Kenneth Reitz**, creator of [`requests`](https://requests.readthedocs.io/en/latest/)\nand [`pipenv`](https://readthedocs.org/projects/pipenv/):\n\n> This vastly improves the formatting of our code. Thanks a ton!\n\n## Show your style\n\nUse the badge in your project\'s README.md:\n\n```md\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n```\n\nUsing the badge in README.rst:\n\n```\n.. image:: https://img.shields.io/badge/code%20style-black-000000.svg\n    :target: https://github.com/psf/black\n```\n\nLooks like this:\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n## License\n\nMIT\n\n## Contributing\n\nWelcome! Happy to see you willing to make the project better. You can get started by\nreading this:\n\n- [Contributing: The basics](https://black.readthedocs.io/en/latest/contributing/the_basics.html)\n\nYou can also take a look at the rest of the contributing docs or talk with the\ndevelopers:\n\n- [Contributing documentation](https://black.readthedocs.io/en/latest/contributing/index.html)\n- [Chat on Discord](https://discord.gg/RtVdv86PrH)\n\n## Change log\n\nThe log has become rather long. It moved to its own file.\n\nSee [CHANGES](https://black.readthedocs.io/en/latest/change_log.html).\n\n## Authors\n\nThe author list is quite long nowadays, so it lives in its own file.\n\nSee [AUTHORS.md](./AUTHORS.md)\n\n## Code of Conduct\n\nEveryone participating in the _Black_ project, and in particular in the issue tracker,\npull requests, and social media activity, is expected to treat other people with respect\nand more generally to follow the guidelines articulated in the\n[Python Community Code of Conduct](https://www.python.org/psf/codeofconduct/).\n\nAt the same time, humor is encouraged. In fact, basic familiarity with Monty Python\'s\nFlying Circus is expected. We are not savages.\n\nAnd if you _really_ need to slap somebody, do it with a fish while dancing.\n# Change Log\n\n## Unreleased\n\n### Highlights\n\n<!-- Include any especially major or disruptive changes here -->\n\n### Stable style\n\n<!-- Changes that affect Black\'s stable style -->\n\n### Preview style\n\n<!-- Changes that affect Black\'s preview style -->\n\n### Configuration\n\n<!-- Changes to how Black can be configured -->\n\n### Packaging\n\n<!-- Changes to how Black is packaged, such as dependency requirements -->\n\n### Parser\n\n<!-- Changes to the parser or to version autodetection -->\n\n### Performance\n\n<!-- Changes that improve Black\'s performance. -->\n\n### Output\n\n<!-- Changes to Black\'s terminal output and error messages -->\n\n### _Blackd_\n\n<!-- Changes to blackd -->\n\n### Integrations\n\n<!-- For example, Docker, GitHub Actions, pre-commit, editors -->\n\n### Documentation\n\n<!-- Major changes to documentation and policies. Small docs changes\n     don\'t need a changelog entry. -->\n\n## 22.10.0\n\n### Highlights\n\n- Runtime support for Python 3.6 has been removed. Formatting 3.6 code will still be\n  supported until further notice.\n\n### Stable style\n\n- Fix a crash when `# fmt: on` is used on a different block level than `# fmt: off`\n  (#3281)\n\n### Preview style\n\n- Fix a crash when formatting some dicts with parenthesis-wrapped long string keys\n  (#3262)\n\n### Configuration\n\n- `.ipynb_checkpoints` directories are now excluded by default (#3293)\n- Add `--skip-source-first-line` / `-x` option to ignore the first line of source code\n  while formatting (#3299)\n\n### Packaging\n\n- Executables made with PyInstaller will no longer crash when formatting several files\n  at once on macOS. Native x86-64 executables for macOS are available once again.\n  (#3275)\n- Hatchling is now used as the build backend. This will not have any effect for users\n  who install Black with its wheels from PyPI. (#3233)\n- Faster compiled wheels are now available for CPython 3.11 (#3276)\n\n### _Blackd_\n\n- Windows style (CRLF) newlines will be preserved (#3257).\n\n### Integrations\n\n- Vim plugin: add flag (`g:black_preview`) to enable/disable the preview style (#3246)\n- Update GitHub Action to support formatting of Jupyter Notebook files via a `jupyter`\n  option (#3282)\n- Update GitHub Action to support use of version specifiers (e.g. `<23`) for Black\n  version (#3265)\n\n## 22.8.0\n\n### Highlights\n\n- Python 3.11 is now supported, except for _blackd_ as aiohttp does not support 3.11 as\n  of publishing (#3234)\n- This is the last release that supports running _Black_ on Python 3.6 (formatting 3.6\n  code will continue to be supported until further notice)\n- Reword the stability policy to say that we may, in rare cases, make changes that\n  affect code that was not previously formatted by _Black_ (#3155)\n\n### Stable style\n\n- Fix an infinite loop when using `# fmt: on/off` in the middle of an expression or code\n  block (#3158)\n- Fix incorrect handling of `# fmt: skip` on colon (`:`) lines (#3148)\n- Comments are no longer deleted when a line had spaces removed around power operators\n  (#2874)\n\n### Preview style\n\n- Single-character closing docstring quotes are no longer moved to their own line as\n  this is invalid. This was a bug introduced in version 22.6.0. (#3166)\n- `--skip-string-normalization` / `-S` now prevents docstring prefixes from being\n  normalized as expected (#3168)\n- When using `--skip-magic-trailing-comma` or `-C`, trailing commas are stripped from\n  subscript expressions with more than 1 element (#3209)\n- Implicitly concatenated strings inside a list, set, or tuple are now wrapped inside\n  parentheses (#3162)\n- Fix a string merging/split issue when a comment is present in the middle of implicitly\n  concatenated strings on its own line (#3227)\n\n### _Blackd_\n\n- `blackd` now supports enabling the preview style via the `X-Preview` header (#3217)\n\n### Configuration\n\n- Black now uses the presence of debug f-strings to detect target version (#3215)\n- Fix misdetection of project root and verbose logging of sources in cases involving\n  `--stdin-filename` (#3216)\n- Immediate `.gitignore` files in source directories given on the command line are now\n  also respected, previously only `.gitignore` files in the project root and\n  automatically discovered directories were respected (#3237)\n\n### Documentation\n\n- Recommend using BlackConnect in IntelliJ IDEs (#3150)\n\n### Integrations\n\n- Vim plugin: prefix messages with `Black: ` so it\'s clear they come from Black (#3194)\n- Docker: changed to a /opt/venv installation + added to PATH to be available to\n  non-root users (#3202)\n\n### Output\n\n- Change from deprecated `asyncio.get_event_loop()` to create our event loop which\n  removes DeprecationWarning (#3164)\n- Remove logging from internal `blib2to3` library since it regularly emits error logs\n  about failed caching that can and should be ignored (#3193)\n\n### Parser\n\n- Type comments are now included in the AST equivalence check consistently so accidental\n  deletion raises an error. Though type comments can\'t be tracked when running on PyPy\n  3.7 due to standard library limitations. (#2874)\n\n### Performance\n\n- Reduce Black\'s startup time when formatting a single file by 15-30% (#3211)\n\n## 22.6.0\n\n### Style\n\n- Fix unstable formatting involving `#fmt: skip` and `# fmt:skip` comments (notice the\n  lack of spaces) (#2970)\n\n### Preview style\n\n- Docstring quotes are no longer moved if it would violate the line length limit (#3044)\n- Parentheses around return annotations are now managed (#2990)\n- Remove unnecessary parentheses around awaited objects (#2991)\n- Remove unnecessary parentheses in `with` statements (#2926)\n- Remove trailing newlines after code block open (#3035)\n\n### Integrations\n\n- Add `scripts/migrate-black.py` script to ease introduction of Black to a Git project\n  (#3038)\n\n### Output\n\n- Output Python version and implementation as part of `--version` flag (#2997)\n\n### Packaging\n\n- Use `tomli` instead of `tomllib` on Python 3.11 builds where `tomllib` is not\n  available (#2987)\n\n### Parser\n\n- [PEP 654](https://peps.python.org/pep-0654/#except) syntax (for example,\n  `except *ExceptionGroup:`) is now supported (#3016)\n- [PEP 646](https://peps.python.org/pep-0646) syntax (for example,\n  `Array[Batch, *Shape]` or `def fn(*args: *T) -> None`) is now supported (#3071)\n\n### Vim Plugin\n\n- Fix `strtobool` function. It didn\'t parse true/on/false/off. (#3025)\n\n## 22.3.0\n\n### Preview style\n\n- Code cell separators `#%%` are now standardised to `# %%` (#2919)\n- Remove unnecessary parentheses from `except` statements (#2939)\n- Remove unnecessary parentheses from tuple unpacking in `for` loops (#2945)\n- Avoid magic-trailing-comma in single-element subscripts (#2942)\n\n### Configuration\n\n- Do not format `__pypackages__` directories by default (#2836)\n- Add support for specifying stable version with `--required-version` (#2832).\n- Avoid crashing when the user has no homedir (#2814)\n- Avoid crashing when md5 is not available (#2905)\n- Fix handling of directory junctions on Windows (#2904)\n\n### Documentation\n\n- Update pylint config documentation (#2931)\n\n### Integrations\n\n- Move test to disable plugin in Vim/Neovim, which speeds up loading (#2896)\n\n### Output\n\n- In verbose mode, log when _Black_ is using user-level config (#2861)\n\n### Packaging\n\n- Fix Black to work with Click 8.1.0 (#2966)\n- On Python 3.11 and newer, use the standard library\'s `tomllib` instead of `tomli`\n  (#2903)\n- `black-primer`, the deprecated internal devtool, has been removed and copied to a\n  [separate repository](https://github.com/cooperlees/black-primer) (#2924)\n\n### Parser\n\n- Black can now parse starred expressions in the target of `for` and `async for`\n  statements, e.g `for item in *items_1, *items_2: pass` (#2879).\n\n## 22.1.0\n\nAt long last, _Black_ is no longer a beta product! This is the first non-beta release\nand the first release covered by our new\n[stability policy](https://black.readthedocs.io/en/stable/the_black_code_style/index.html#stability-policy).\n\n### Highlights\n\n- **Remove Python 2 support** (#2740)\n- Introduce the `--preview` flag (#2752)\n\n### Style\n\n- Deprecate `--experimental-string-processing` and move the functionality under\n  `--preview` (#2789)\n- For stubs, one blank line between class attributes and methods is now kept if there\'s\n  at least one pre-existing blank line (#2736)\n- Black now normalizes string prefix order (#2297)\n- Remove spaces around power operators if both operands are simple (#2726)\n- Work around bug that causes unstable formatting in some cases in the presence of the\n  magic trailing comma (#2807)\n- Use parentheses for attribute access on decimal float and int literals (#2799)\n- Don\'t add whitespace for attribute access on hexadecimal, binary, octal, and complex\n  literals (#2799)\n- Treat blank lines in stubs the same inside top-level `if` statements (#2820)\n- Fix unstable formatting with semicolons and arithmetic expressions (#2817)\n- Fix unstable formatting around magic trailing comma (#2572)\n\n### Parser\n\n- Fix mapping cases that contain as-expressions, like `case {"key": 1 | 2 as password}`\n  (#2686)\n- Fix cases that contain multiple top-level as-expressions, like `case 1 as a, 2 as b`\n  (#2716)\n- Fix call patterns that contain as-expressions with keyword arguments, like\n  `case Foo(bar=baz as quux)` (#2749)\n- Tuple unpacking on `return` and `yield` constructs now implies 3.8+ (#2700)\n- Unparenthesized tuples on annotated assignments (e.g\n  `values: Tuple[int, ...] = 1, 2, 3`) now implies 3.8+ (#2708)\n- Fix handling of standalone `match()` or `case()` when there is a trailing newline or a\n  comment inside of the parentheses. (#2760)\n- `from __future__ import annotations` statement now implies Python 3.7+ (#2690)\n\n### Performance\n\n- Speed-up the new backtracking parser about 4X in general (enabled when\n  `--target-version` is set to 3.10 and higher). (#2728)\n- _Black_ is now compiled with [mypyc](https://github.com/mypyc/mypyc) for an overall 2x\n  speed-up. 64-bit Windows, MacOS, and Linux (not including musl) are supported. (#1009,\n  #2431)\n\n### Configuration\n\n- Do not accept bare carriage return line endings in pyproject.toml (#2408)\n- Add configuration option (`python-cell-magics`) to format cells with custom magics in\n  Jupyter Notebooks (#2744)\n- Allow setting custom cache directory on all platforms with environment variable\n  `BLACK_CACHE_DIR` (#2739).\n- Enable Python 3.10+ by default, without any extra need to specify\n  `--target-version=py310`. (#2758)\n- Make passing `SRC` or `--code` mandatory and mutually exclusive (#2804)\n\n### Output\n\n- Improve error message for invalid regular expression (#2678)\n- Improve error message when parsing fails during AST safety check by embedding the\n  underlying SyntaxError (#2693)\n- No longer color diff headers white as it\'s unreadable in light themed terminals\n  (#2691)\n- Text coloring added in the final statistics (#2712)\n- Verbose mode also now describes how a project root was discovered and which paths will\n  be formatted. (#2526)\n\n### Packaging\n\n- All upper version bounds on dependencies have been removed (#2718)\n- `typing-extensions` is no longer a required dependency in Python 3.10+ (#2772)\n- Set `click` lower bound to `8.0.0` (#2791)\n\n### Integrations\n\n- Update GitHub action to support containerized runs (#2748)\n\n### Documentation\n\n- Change protocol in pip installation instructions to `https://` (#2761)\n- Change HTML theme to Furo primarily for its responsive design and mobile support\n  (#2793)\n- Deprecate the `black-primer` tool (#2809)\n- Document Python support policy (#2819)\n\n## 21.12b0\n\n### _Black_\n\n- Fix determination of f-string expression spans (#2654)\n- Fix bad formatting of error messages about EOF in multi-line statements (#2343)\n- Functions and classes in blocks now have more consistent surrounding spacing (#2472)\n\n#### Jupyter Notebook support\n\n- Cell magics are now only processed if they are known Python cell magics. Earlier, all\n  cell magics were tokenized, leading to possible indentation errors e.g. with\n  `%%writefile`. (#2630)\n- Fix assignment to environment variables in Jupyter Notebooks (#2642)\n\n#### Python 3.10 support\n\n- Point users to using `--target-version py310` if we detect 3.10-only syntax (#2668)\n- Fix `match` statements with open sequence subjects, like `match a, b:` or\n  `match a, *b:` (#2639) (#2659)\n- Fix `match`/`case` statements that contain `match`/`case` soft keywords multiple\n  times, like `match re.match()` (#2661)\n- Fix `case` statements with an inline body (#2665)\n- Fix styling of starred expressions inside `match` subject (#2667)\n- Fix parser error location on invalid syntax in a `match` statement (#2649)\n- Fix Python 3.10 support on platforms without ProcessPoolExecutor (#2631)\n- Improve parsing performance on code that uses `match` under `--target-version py310`\n  up to ~50% (#2670)\n\n### Packaging\n\n- Remove dependency on `regex` (#2644) (#2663)\n\n## 21.11b1\n\n### _Black_\n\n- Bumped regex version minimum to 2021.4.4 to fix Pattern class usage (#2621)\n\n## 21.11b0\n\n### _Black_\n\n- Warn about Python 2 deprecation in more cases by improving Python 2 only syntax\n  detection (#2592)\n- Add experimental PyPy support (#2559)\n- Add partial support for the match statement. As it\'s experimental, it\'s only enabled\n  when `--target-version py310` is explicitly specified (#2586)\n- Add support for parenthesized with (#2586)\n- Declare support for Python 3.10 for running Black (#2562)\n\n### Integrations\n\n- Fixed vim plugin with Python 3.10 by removing deprecated distutils import (#2610)\n- The vim plugin now parses `skip_magic_trailing_comma` from pyproject.toml (#2613)\n\n## 21.10b0\n\n### _Black_\n\n- Document stability policy, that will apply for non-beta releases (#2529)\n- Add new `--workers` parameter (#2514)\n- Fixed feature detection for positional-only arguments in lambdas (#2532)\n- Bumped typed-ast version minimum to 1.4.3 for 3.10 compatibility (#2519)\n- Fixed a Python 3.10 compatibility issue where the loop argument was still being passed\n  even though it has been removed (#2580)\n- Deprecate Python 2 formatting support (#2523)\n\n### _Blackd_\n\n- Remove dependency on aiohttp-cors (#2500)\n- Bump required aiohttp version to 3.7.4 (#2509)\n\n### _Black-Primer_\n\n- Add primer support for --projects (#2555)\n- Print primer summary after individual failures (#2570)\n\n### Integrations\n\n- Allow to pass `target_version` in the vim plugin (#1319)\n- Install build tools in docker file and use multi-stage build to keep the image size\n  down (#2582)\n\n## 21.9b0\n\n### Packaging\n\n- Fix missing modules in self-contained binaries (#2466)\n- Fix missing toml extra used during installation (#2475)\n\n## 21.8b0\n\n### _Black_\n\n- Add support for formatting Jupyter Notebook files (#2357)\n- Move from `appdirs` dependency to `platformdirs` (#2375)\n- Present a more user-friendly error if .gitignore is invalid (#2414)\n- The failsafe for accidentally added backslashes in f-string expressions has been\n  hardened to handle more edge cases during quote normalization (#2437)\n- Avoid changing a function return type annotation\'s type to a tuple by adding a\n  trailing comma (#2384)\n- Parsing support has been added for unparenthesized walruses in set literals, set\n  comprehensions, and indices (#2447).\n- Pin `setuptools-scm` build-time dependency version (#2457)\n- Exclude typing-extensions version 3.10.0.1 due to it being broken on Python 3.10\n  (#2460)\n\n### _Blackd_\n\n- Replace sys.exit(-1) with raise ImportError as it plays more nicely with tools that\n  scan installed packages (#2440)\n\n### Integrations\n\n- The provided pre-commit hooks no longer specify `language_version` to avoid overriding\n  `default_language_version` (#2430)\n\n## 21.7b0\n\n### _Black_\n\n- Configuration files using TOML features higher than spec v0.5.0 are now supported\n  (#2301)\n- Add primer support and test for code piped into black via STDIN (#2315)\n- Fix internal error when `FORCE_OPTIONAL_PARENTHESES` feature is enabled (#2332)\n- Accept empty stdin (#2346)\n- Provide a more useful error when parsing fails during AST safety checks (#2304)\n\n### Docker\n\n- Add new `latest_release` tag automation to follow latest black release on docker\n  images (#2374)\n\n### Integrations\n\n- The vim plugin now searches upwards from the directory containing the current buffer\n  instead of the current working directory for pyproject.toml. (#1871)\n- The vim plugin now reads the correct string normalization option in pyproject.toml\n  (#1869)\n- The vim plugin no longer crashes Black when there\'s boolean values in pyproject.toml\n  (#1869)\n\n## 21.6b0\n\n### _Black_\n\n- Fix failure caused by `fmt: skip` and indentation (#2281)\n- Account for += assignment when deciding whether to split string (#2312)\n- Correct max string length calculation when there are string operators (#2292)\n- Fixed option usage when using the `--code` flag (#2259)\n- Do not call `uvloop.install()` when _Black_ is used as a library (#2303)\n- Added `--required-version` option to require a specific version to be running (#2300)\n- Fix incorrect custom breakpoint indices when string group contains fake f-strings\n  (#2311)\n- Fix regression where `R` prefixes would be lowercased for docstrings (#2285)\n- Fix handling of named escapes (`\\N{...}`) when `--experimental-string-processing` is\n  used (#2319)\n\n### Integrations\n\n- The official Black action now supports choosing what version to use, and supports the\n  major 3 OSes. (#1940)\n\n## 21.5b2\n\n### _Black_\n\n- A space is no longer inserted into empty docstrings (#2249)\n- Fix handling of .gitignore files containing non-ASCII characters on Windows (#2229)\n- Respect `.gitignore` files in all levels, not only `root/.gitignore` file (apply\n  `.gitignore` rules like `git` does) (#2225)\n- Restored compatibility with Click 8.0 on Python 3.6 when LANG=C used (#2227)\n- Add extra uvloop install + import support if in python env (#2258)\n- Fix --experimental-string-processing crash when matching parens are not found (#2283)\n- Make sure to split lines that start with a string operator (#2286)\n- Fix regular expression that black uses to identify f-expressions (#2287)\n\n### _Blackd_\n\n- Add a lower bound for the `aiohttp-cors` dependency. Only 0.4.0 or higher is\n  supported. (#2231)\n\n### Packaging\n\n- Release self-contained x86_64 MacOS binaries as part of the GitHub release pipeline\n  (#2198)\n- Always build binaries with the latest available Python (#2260)\n\n### Documentation\n\n- Add discussion of magic comments to FAQ page (#2272)\n- `--experimental-string-processing` will be enabled by default in the future (#2273)\n- Fix typos discovered by codespell (#2228)\n- Fix Vim plugin installation instructions. (#2235)\n- Add new Frequently Asked Questions page (#2247)\n- Fix encoding + symlink issues preventing proper build on Windows (#2262)\n\n## 21.5b1\n\n### _Black_\n\n- Refactor `src/black/__init__.py` into many files (#2206)\n\n### Documentation\n\n- Replaced all remaining references to the\n  [`master`](https://github.com/psf/black/tree/main) branch with the\n  [`main`](https://github.com/psf/black/tree/main) branch. Some additional changes in\n  the source code were also made. (#2210)\n- Sigificantly reorganized the documentation to make much more sense. Check them out by\n  heading over to [the stable docs on RTD](https://black.readthedocs.io/en/stable/).\n  (#2174)\n\n## 21.5b0\n\n### _Black_\n\n- Set `--pyi` mode if `--stdin-filename` ends in `.pyi` (#2169)\n- Stop detecting target version as Python 3.9+ with pre-PEP-614 decorators that are\n  being called but with no arguments (#2182)\n\n### _Black-Primer_\n\n- Add `--no-diff` to black-primer to suppress formatting changes (#2187)\n\n## 21.4b2\n\n### _Black_\n\n- Fix crash if the user configuration directory is inaccessible. (#2158)\n\n- Clarify\n  [circumstances](https://github.com/psf/black/blob/master/docs/the_black_code_style.md#pragmatism)\n  in which _Black_ may change the AST (#2159)\n\n- Allow `.gitignore` rules to be overridden by specifying `exclude` in `pyproject.toml`\n  or on the command line. (#2170)\n\n### _Packaging_\n\n- Install `primer.json` (used by `black-primer` by default) with black. (#2154)\n\n## 21.4b1\n\n### _Black_\n\n- Fix crash on docstrings ending with "\\\\ ". (#2142)\n\n- Fix crash when atypical whitespace is cleaned out of dostrings (#2120)\n\n- Reflect the `--skip-magic-trailing-comma` and `--experimental-string-processing` flags\n  in the name of the cache file. Without this fix, changes in these flags would not take\n  effect if the cache had already been populated. (#2131)\n\n- Don\'t remove necessary parentheses from assignment expression containing assert /\n  return statements. (#2143)\n\n### _Packaging_\n\n- Bump pathspec to >= 0.8.1 to solve invalid .gitignore exclusion handling\n\n## 21.4b0\n\n### _Black_\n\n- Fixed a rare but annoying formatting instability created by the combination of\n  optional trailing commas inserted by `Black` and optional parentheses looking at\n  pre-existing "magic" trailing commas. This fixes issue #1629 and all of its many many\n  duplicates. (#2126)\n\n- `Black` now processes one-line docstrings by stripping leading and trailing spaces,\n  and adding a padding space when needed to break up """". (#1740)\n\n- `Black` now cleans up leading non-breaking spaces in comments (#2092)\n\n- `Black` now respects `--skip-string-normalization` when normalizing multiline\n  docstring quotes (#1637)\n\n- `Black` no longer removes all empty lines between non-function code and decorators\n  when formatting typing stubs. Now `Black` enforces a single empty line. (#1646)\n\n- `Black` no longer adds an incorrect space after a parenthesized assignment expression\n  in if/while statements (#1655)\n\n- Added `--skip-magic-trailing-comma` / `-C` to avoid using trailing commas as a reason\n  to split lines (#1824)\n\n- fixed a crash when PWD=/ on POSIX (#1631)\n\n- fixed "I/O operation on closed file" when using --diff (#1664)\n\n- Prevent coloured diff output being interleaved with multiple files (#1673)\n\n- Added support for PEP 614 relaxed decorator syntax on python 3.9 (#1711)\n\n- Added parsing support for unparenthesized tuples and yield expressions in annotated\n  assignments (#1835)\n\n- added `--extend-exclude` argument (PR #2005)\n\n- speed up caching by avoiding pathlib (#1950)\n\n- `--diff` correctly indicates when a file doesn\'t end in a newline (#1662)\n\n- Added `--stdin-filename` argument to allow stdin to respect `--force-exclude` rules\n  (#1780)\n\n- Lines ending with `fmt: skip` will now be not formatted (#1800)\n\n- PR #2053: Black no longer relies on typed-ast for Python 3.8 and higher\n\n- PR #2053: Python 2 support is now optional, install with\n  `python3 -m pip install black[python2]` to maintain support.\n\n- Exclude `venv` directory by default (#1683)\n\n- Fixed "Black produced code that is not equivalent to the source" when formatting\n  Python 2 docstrings (#2037)\n\n### _Packaging_\n\n- Self-contained native _Black_ binaries are now provided for releases via GitHub\n  Releases (#1743)\n\n## 20.8b1\n\n### _Packaging_\n\n- explicitly depend on Click 7.1.2 or newer as `Black` no longer works with versions\n  older than 7.0\n\n## 20.8b0\n\n### _Black_\n\n- re-implemented support for explicit trailing commas: now it works consistently within\n  any bracket pair, including nested structures (#1288 and duplicates)\n\n- `Black` now reindents docstrings when reindenting code around it (#1053)\n\n- `Black` now shows colored diffs (#1266)\n\n- `Black` is now packaged using \'py3\' tagged wheels (#1388)\n\n- `Black` now supports Python 3.8 code, e.g. star expressions in return statements\n  (#1121)\n\n- `Black` no longer normalizes capital R-string prefixes as those have a\n  community-accepted meaning (#1244)\n\n- `Black` now uses exit code 2 when specified configuration file doesn\'t exit (#1361)\n\n- `Black` now works on AWS Lambda (#1141)\n\n- added `--force-exclude` argument (#1032)\n\n- removed deprecated `--py36` option (#1236)\n\n- fixed `--diff` output when EOF is encountered (#526)\n\n- fixed `# fmt: off` handling around decorators (#560)\n\n- fixed unstable formatting with some `# type: ignore` comments (#1113)\n\n- fixed invalid removal on organizing brackets followed by indexing (#1575)\n\n- introduced `black-primer`, a CI tool that allows us to run regression tests against\n  existing open source users of Black (#1402)\n\n- introduced property-based fuzzing to our test suite based on Hypothesis and\n  Hypothersmith (#1566)\n\n- implemented experimental and disabled by default long string rewrapping (#1132),\n  hidden under a `--experimental-string-processing` flag while it\'s being worked on;\n  this is an undocumented and unsupported feature, you lose Internet points for\n  depending on it (#1609)\n\n### Vim plugin\n\n- prefer virtualenv packages over global packages (#1383)\n\n## 19.10b0\n\n- added support for PEP 572 assignment expressions (#711)\n\n- added support for PEP 570 positional-only arguments (#943)\n\n- added support for async generators (#593)\n\n- added support for pre-splitting collections by putting an explicit trailing comma\n  inside (#826)\n\n- added `black -c` as a way to format code passed from the command line (#761)\n\n- --safe now works with Python 2 code (#840)\n\n- fixed grammar selection for Python 2-specific code (#765)\n\n- fixed feature detection for trailing commas in function definitions and call sites\n  (#763)\n\n- `# fmt: off`/`# fmt: on` comment pairs placed multiple times within the same block of\n  code now behave correctly (#1005)\n\n- _Black_ no longer crashes on Windows machines with more than 61 cores (#838)\n\n- _Black_ no longer crashes on standalone comments prepended with a backslash (#767)\n\n- _Black_ no longer crashes on `from` ... `import` blocks with comments (#829)\n\n- _Black_ no longer crashes on Python 3.7 on some platform configurations (#494)\n\n- _Black_ no longer fails on comments in from-imports (#671)\n\n- _Black_ no longer fails when the file starts with a backslash (#922)\n\n- _Black_ no longer merges regular comments with type comments (#1027)\n\n- _Black_ no longer splits long lines that contain type comments (#997)\n\n- removed unnecessary parentheses around `yield` expressions (#834)\n\n- added parentheses around long tuples in unpacking assignments (#832)\n\n- added parentheses around complex powers when they are prefixed by a unary operator\n  (#646)\n\n- fixed bug that led _Black_ format some code with a line length target of 1 (#762)\n\n- _Black_ no longer introduces quotes in f-string subexpressions on string boundaries\n  (#863)\n\n- if _Black_ puts parenthesis around a single expression, it moves comments to the\n  wrapped expression instead of after the brackets (#872)\n\n- `blackd` now returns the version of _Black_ in the response headers (#1013)\n\n- `blackd` can now output the diff of formats on source code when the `X-Diff` header is\n  provided (#969)\n\n## 19.3b0\n\n- new option `--target-version` to control which Python versions _Black_-formatted code\n  should target (#618)\n\n- deprecated `--py36` (use `--target-version=py36` instead) (#724)\n\n- _Black_ no longer normalizes numeric literals to include `_` separators (#696)\n\n- long `del` statements are now split into multiple lines (#698)\n\n- type comments are no longer mangled in function signatures\n\n- improved performance of formatting deeply nested data structures (#509)\n\n- _Black_ now properly formats multiple files in parallel on Windows (#632)\n\n- _Black_ now creates cache files atomically which allows it to be used in parallel\n  pipelines (like `xargs -P8`) (#673)\n\n- _Black_ now correctly indents comments in files that were previously formatted with\n  tabs (#262)\n\n- `blackd` now supports CORS (#622)\n\n## 18.9b0\n\n- numeric literals are now formatted by _Black_ (#452, #461, #464, #469):\n\n  - numeric literals are normalized to include `_` separators on Python 3.6+ code\n\n  - added `--skip-numeric-underscore-normalization` to disable the above behavior and\n    leave numeric underscores as they were in the input\n\n  - code with `_` in numeric literals is recognized as Python 3.6+\n\n  - most letters in numeric literals are lowercased (e.g., in `1e10`, `0x01`)\n\n  - hexadecimal digits are always uppercased (e.g. `0xBADC0DE`)\n\n- added `blackd`, see\n  [its documentation](https://github.com/psf/black/blob/18.9b0/README.md#blackd) for\n  more info (#349)\n\n- adjacent string literals are now correctly split into multiple lines (#463)\n\n- trailing comma is now added to single imports that don\'t fit on a line (#250)\n\n- cache is now populated when `--check` is successful for a file which speeds up\n  consecutive checks of properly formatted unmodified files (#448)\n\n- whitespace at the beginning of the file is now removed (#399)\n\n- fixed mangling [pweave](http://mpastell.com/pweave/) and\n  [Spyder IDE](https://www.spyder-ide.org/) special comments (#532)\n\n- fixed unstable formatting when unpacking big tuples (#267)\n\n- fixed parsing of `__future__` imports with renames (#389)\n\n- fixed scope of `# fmt: off` when directly preceding `yield` and other nodes (#385)\n\n- fixed formatting of lambda expressions with default arguments (#468)\n\n- fixed `async for` statements: _Black_ no longer breaks them into separate lines (#372)\n\n- note: the Vim plugin stopped registering `,=` as a default chord as it turned out to\n  be a bad idea (#415)\n\n## 18.6b4\n\n- hotfix: don\'t freeze when multiple comments directly precede `# fmt: off` (#371)\n\n## 18.6b3\n\n- typing stub files (`.pyi`) now have blank lines added after constants (#340)\n\n- `# fmt: off` and `# fmt: on` are now much more dependable:\n\n  - they now work also within bracket pairs (#329)\n\n  - they now correctly work across function/class boundaries (#335)\n\n  - they now work when an indentation block starts with empty lines or misaligned\n    comments (#334)\n\n- made Click not fail on invalid environments; note that Click is right but the\n  likelihood we\'ll need to access non-ASCII file paths when dealing with Python source\n  code is low (#277)\n\n- fixed improper formatting of f-strings with quotes inside interpolated expressions\n  (#322)\n\n- fixed unnecessary slowdown when long list literals where found in a file\n\n- fixed unnecessary slowdown on AST nodes with very many siblings\n\n- fixed cannibalizing backslashes during string normalization\n\n- fixed a crash due to symbolic links pointing outside of the project directory (#338)\n\n## 18.6b2\n\n- added `--config` (#65)\n\n- added `-h` equivalent to `--help` (#316)\n\n- fixed improper unmodified file caching when `-S` was used\n\n- fixed extra space in string unpacking (#305)\n\n- fixed formatting of empty triple quoted strings (#313)\n\n- fixed unnecessary slowdown in comment placement calculation on lines without comments\n\n## 18.6b1\n\n- hotfix: don\'t output human-facing information on stdout (#299)\n\n- hotfix: don\'t output cake emoji on non-zero return code (#300)\n\n## 18.6b0\n\n- added `--include` and `--exclude` (#270)\n\n- added `--skip-string-normalization` (#118)\n\n- added `--verbose` (#283)\n\n- the header output in `--diff` now actually conforms to the unified diff spec\n\n- fixed long trivial assignments being wrapped in unnecessary parentheses (#273)\n\n- fixed unnecessary parentheses when a line contained multiline strings (#232)\n\n- fixed stdin handling not working correctly if an old version of Click was used (#276)\n\n- _Black_ now preserves line endings when formatting a file in place (#258)\n\n## 18.5b1\n\n- added `--pyi` (#249)\n\n- added `--py36` (#249)\n\n- Python grammar pickle caches are stored with the formatting caches, making _Black_\n  work in environments where site-packages is not user-writable (#192)\n\n- _Black_ now enforces a PEP 257 empty line after a class-level docstring (and/or\n  fields) and the first method\n\n- fixed invalid code produced when standalone comments were present in a trailer that\n  was omitted from line splitting on a large expression (#237)\n\n- fixed optional parentheses being removed within `# fmt: off` sections (#224)\n\n- fixed invalid code produced when stars in very long imports were incorrectly wrapped\n  in optional parentheses (#234)\n\n- fixed unstable formatting when inline comments were moved around in a trailer that was\n  omitted from line splitting on a large expression (#238)\n\n- fixed extra empty line between a class declaration and the first method if no class\n  docstring or fields are present (#219)\n\n- fixed extra empty line between a function signature and an inner function or inner\n  class (#196)\n\n## 18.5b0\n\n- call chains are now formatted according to the\n  [fluent interfaces](https://en.wikipedia.org/wiki/Fluent_interface) style (#67)\n\n- data structure literals (tuples, lists, dictionaries, and sets) are now also always\n  exploded like imports when they don\'t fit in a single line (#152)\n\n- slices are now formatted according to PEP 8 (#178)\n\n- parentheses are now also managed automatically on the right-hand side of assignments\n  and return statements (#140)\n\n- math operators now use their respective priorities for delimiting multiline\n  expressions (#148)\n\n- optional parentheses are now omitted on expressions that start or end with a bracket\n  and only contain a single operator (#177)\n\n- empty parentheses in a class definition are now removed (#145, #180)\n\n- string prefixes are now standardized to lowercase and `u` is removed on Python 3.6+\n  only code and Python 2.7+ code with the `unicode_literals` future import (#188, #198,\n  #199)\n\n- typing stub files (`.pyi`) are now formatted in a style that is consistent with PEP\n  484 (#207, #210)\n\n- progress when reformatting many files is now reported incrementally\n\n- fixed trailers (content with brackets) being unnecessarily exploded into their own\n  lines after a dedented closing bracket (#119)\n\n- fixed an invalid trailing comma sometimes left in imports (#185)\n\n- fixed non-deterministic formatting when multiple pairs of removable parentheses were\n  used (#183)\n\n- fixed multiline strings being unnecessarily wrapped in optional parentheses in long\n  assignments (#215)\n\n- fixed not splitting long from-imports with only a single name\n\n- fixed Python 3.6+ file discovery by also looking at function calls with unpacking.\n  This fixed non-deterministic formatting if trailing commas where used both in function\n  signatures with stars and function calls with stars but the former would be\n  reformatted to a single line.\n\n- fixed crash on dealing with optional parentheses (#193)\n\n- fixed "is", "is not", "in", and "not in" not considered operators for splitting\n  purposes\n\n- fixed crash when dead symlinks where encountered\n\n## 18.4a4\n\n- don\'t populate the cache on `--check` (#175)\n\n## 18.4a3\n\n- added a "cache"; files already reformatted that haven\'t changed on disk won\'t be\n  reformatted again (#109)\n\n- `--check` and `--diff` are no longer mutually exclusive (#149)\n\n- generalized star expression handling, including double stars; this fixes\n  multiplication making expressions "unsafe" for trailing commas (#132)\n\n- _Black_ no longer enforces putting empty lines behind control flow statements (#90)\n\n- _Black_ now splits imports like "Mode 3 + trailing comma" of isort (#127)\n\n- fixed comment indentation when a standalone comment closes a block (#16, #32)\n\n- fixed standalone comments receiving extra empty lines if immediately preceding a\n  class, def, or decorator (#56, #154)\n\n- fixed `--diff` not showing entire path (#130)\n\n- fixed parsing of complex expressions after star and double stars in function calls\n  (#2)\n\n- fixed invalid splitting on comma in lambda arguments (#133)\n\n- fixed missing splits of ternary expressions (#141)\n\n## 18.4a2\n\n- fixed parsing of unaligned standalone comments (#99, #112)\n\n- fixed placement of dictionary unpacking inside dictionary literals (#111)\n\n- Vim plugin now works on Windows, too\n\n- fixed unstable formatting when encountering unnecessarily escaped quotes in a string\n  (#120)\n\n## 18.4a1\n\n- added `--quiet` (#78)\n\n- added automatic parentheses management (#4)\n\n- added [pre-commit](https://pre-commit.com) integration (#103, #104)\n\n- fixed reporting on `--check` with multiple files (#101, #102)\n\n- fixed removing backslash escapes from raw strings (#100, #105)\n\n## 18.4a0\n\n- added `--diff` (#87)\n\n- add line breaks before all delimiters, except in cases like commas, to better comply\n  with PEP 8 (#73)\n\n- standardize string literals to use double quotes (almost) everywhere (#75)\n\n- fixed handling of standalone comments within nested bracketed expressions; _Black_\n  will no longer produce super long lines or put all standalone comments at the end of\n  the expression (#22)\n\n- fixed 18.3a4 regression: don\'t crash and burn on empty lines with trailing whitespace\n  (#80)\n\n- fixed 18.3a4 regression: `# yapf: disable` usage as trailing comment would cause\n  _Black_ to not emit the rest of the file (#95)\n\n- when CTRL+C is pressed while formatting many files, _Black_ no longer freaks out with\n  a flurry of asyncio-related exceptions\n\n- only allow up to two empty lines on module level and only single empty lines within\n  functions (#74)\n\n## 18.3a4\n\n- `# fmt: off` and `# fmt: on` are implemented (#5)\n\n- automatic detection of deprecated Python 2 forms of print statements and exec\n  statements in the formatted file (#49)\n\n- use proper spaces for complex expressions in default values of typed function\n  arguments (#60)\n\n- only return exit code 1 when --check is used (#50)\n\n- don\'t remove single trailing commas from square bracket indexing (#59)\n\n- don\'t omit whitespace if the previous factor leaf wasn\'t a math operator (#55)\n\n- omit extra space in kwarg unpacking if it\'s the first argument (#46)\n\n- omit extra space in\n  [Sphinx auto-attribute comments](http://www.sphinx-doc.org/en/stable/ext/autodoc.html#directive-autoattribute)\n  (#68)\n\n## 18.3a3\n\n- don\'t remove single empty lines outside of bracketed expressions (#19)\n\n- added ability to pipe formatting from stdin to stdin (#25)\n\n- restored ability to format code with legacy usage of `async` as a name (#20, #42)\n\n- even better handling of numpy-style array indexing (#33, again)\n\n## 18.3a2\n\n- changed positioning of binary operators to occur at beginning of lines instead of at\n  the end, following\n  [a recent change to PEP 8](https://github.com/python/peps/commit/c59c4376ad233a62ca4b3a6060c81368bd21e85b)\n  (#21)\n\n- ignore empty bracket pairs while splitting. This avoids very weirdly looking\n  formattings (#34, #35)\n\n- remove a trailing comma if there is a single argument to a call\n\n- if top level functions were separated by a comment, don\'t put four empty lines after\n  the upper function\n\n- fixed unstable formatting of newlines with imports\n\n- fixed unintentional folding of post scriptum standalone comments into last statement\n  if it was a simple statement (#18, #28)\n\n- fixed missing space in numpy-style array indexing (#33)\n\n- fixed spurious space after star-based unary expressions (#31)\n\n## 18.3a1\n\n- added `--check`\n\n- only put trailing commas in function signatures and calls if it\'s safe to do so. If\n  the file is Python 3.6+ it\'s always safe, otherwise only safe if there are no `*args`\n  or `**kwargs` used in the signature or call. (#8)\n\n- fixed invalid spacing of dots in relative imports (#6, #13)\n\n- fixed invalid splitting after comma on unpacked variables in for-loops (#23)\n\n- fixed spurious space in parenthesized set expressions (#7)\n\n- fixed spurious space after opening parentheses and in default arguments (#14, #17)\n\n- fixed spurious space after unary operators when the operand was a complex expression\n  (#15)\n\n## 18.3a0\n\n- first published version, Happy 🍰 Day 2018!\n\n- alpha quality\n\n- date-versioned (see: <https://calver.org/>)\n',
    author_email='Łukasz Langa <lukasz@langa.pl>',
    classifiers=[
        'Development Status :: 5 - Production/Stable',
        'Environment :: Console',
        'Intended Audience :: Developers',
        'License :: OSI Approved :: MIT License',
        'Operating System :: OS Independent',
        'Programming Language :: Python',
        'Programming Language :: Python :: 3 :: Only',
        'Programming Language :: Python :: 3.7',
        'Programming Language :: Python :: 3.8',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: 3.10',
        'Programming Language :: Python :: 3.11',
        'Topic :: Software Development :: Libraries :: Python Modules',
        'Topic :: Software Development :: Quality Assurance',
    ],
    install_requires=[
        'click>=8.0.0',
        'mypy-extensions>=0.4.3',
        'pathspec>=0.9.0',
        'platformdirs>=2',
        'tomli>=1.1.0; python_full_version < "3.11.0a7"',
        'typed-ast>=1.4.2; python_version < "3.8" and implementation_name == "cpython"',
        'typing-extensions>=3.10.0.0; python_version < "3.10"',
    ],
    extras_require={
        'colorama': [
            'colorama>=0.4.3',
        ],
        'd': [
            'aiohttp>=3.7.4',
        ],
        'jupyter': [
            'ipython>=7.8.0',
            'tokenize-rt>=3.2.0',
        ],
        'uvloop': [
            'uvloop>=0.15.2',
        ],
    },
    entry_points={
        'console_scripts': [
            'black = black:patched_main',
            'blackd = blackd:patched_main [d]',
        ],
    },
    package_dir={
        '': 'src',
    },
    package_data={
        '': ['*'],
    },
)
