# Pass Infrastructure

[TOC]

Passes represent the basic infrastructure for transformation and optimization.
This document provides an overview of the pass infrastructure in MLIR and how to
use it.

See [MLIR specification](LangRef.md) for more information about MLIR and its
core aspects, such as the IR structure and operations.

See [MLIR Rewrites](Tutorials/QuickstartRewrites.md) for a quick start on graph
rewriting in MLIR. If a transformation involves pattern matching operation DAGs,
this is a great place to start.

## Operation Pass

In MLIR, the main unit of abstraction and transformation is an
[operation](LangRef.md#operations). As such, the pass manager is designed to
work on instances of operations at different levels of nesting. The structure of
the [pass manager](#pass-manager), and the concept of nesting, is detailed
further below. All passes in MLIR derive from `OperationPass` and adhere to the
following restrictions; any noncompliance will lead to problematic behavior in
multithreaded and other advanced scenarios:

*   Modify any state referenced or relied upon outside the current being
    operated on. This includes adding or removing operations from the parent
    block, changing the attributes(depending on the contract of the current
    operation)/operands/results/successors of the current operation.
*   Modify the state of another operation not nested within the current
    operation being operated on.
    *   Other threads may be operating on these operations simultaneously.
*   Inspect the state of sibling operations.
    *   Other threads may be modifying these operations in parallel.
    *   Inspecting the state of ancestor/parent operations is permitted.
*   Maintain mutable pass state across invocations of `runOnOperation`. A pass
    may be run on many different operations with no guarantee of execution
    order.
    *   When multithreading, a specific pass instance may not even execute on
        all operations within the IR. As such, a pass should not rely on running
        on all operations.
*   Maintain any global mutable state, e.g. static variables within the source
    file. All mutable state should be maintained by an instance of the pass.
*   Must be copy-constructible
    *   Multiple instances of the pass may be created by the pass manager to
        process operations in parallel.

When creating an operation pass, there are two different types to choose from
depending on the usage scenario:

### OperationPass : Op-Specific

An `op-specific` operation pass operates explicitly on a given operation type.
This operation type must adhere to the restrictions set by the pass manager for
pass execution.

To define an op-specific operation pass, a derived class must adhere to the
following:

*   Inherit from the CRTP class `OperationPass` and provide the operation type
    as an additional template parameter.
*   Override the virtual `void runOnOperation()` method.

A simple pass may look like:

```c++
namespace {
/// Here we utilize the CRTP `PassWrapper` utility class to provide some
/// necessary utility hooks. This is only necessary for passes defined directly
/// in C++. Passes defined declaratively use a cleaner mechanism for providing
/// these utilities.
struct MyFunctionPass : public PassWrapper<OperationPass<FuncOp>,
                                           MyFunctionPass> {
  void runOnOperation() override {
    // Get the current FuncOp operation being operated on.
    FuncOp f = getOperation();

    // Walk the operations within the function.
    f.walk([](Operation *inst) {
      ....
    });
  }
};
} // end anonymous namespace

/// Register this pass so that it can be built via from a textual pass pipeline.
/// (Pass registration is discussed more below)
void registerMyPass() {
  PassRegistration<MyFunctionPass>(
    "flag-name-to-invoke-pass-via-mlir-opt", "Pass description here");
}
```

### OperationPass : Op-Agnostic

An `op-agnostic` pass operates on the operation type of the pass manager that it
is added to. This means that passes of this type may operate on several
different operation types. Passes of this type are generally written generically
using operation [interfaces](Interfaces.md) and [traits](Traits.md). Examples of
this type of pass are
[Common Sub-Expression Elimination](Passes.md#-cse-eliminate-common-sub-expressions)
and [Inlining](Passes.md#-inline-inline-function-calls).

To create an operation pass, a derived class must adhere to the following:

*   Inherit from the CRTP class `OperationPass`.
*   Override the virtual `void runOnOperation()` method.

A simple pass may look like:

```c++
/// Here we utilize the CRTP `PassWrapper` utility class to provide some
/// necessary utility hooks. This is only necessary for passes defined directly
/// in C++. Passes defined declaratively use a cleaner mechanism for providing
/// these utilities.
struct MyOperationPass : public PassWrapper<OperationPass<>, MyOperationPass> {
  void runOnOperation() override {
    // Get the current operation being operated on.
    Operation *op = getOperation();
    ...
  }
};
```

### Dependent Dialects

Dialects must be loaded in the MLIRContext before entities from these dialects
(operations, types, attributes, ...) can be created. Dialects must also be
loaded before starting the execution of a multi-threaded pass pipeline. To this
end, a pass that may create an entity from a dialect that isn't guaranteed to
already ne loaded must express this by overriding the `getDependentDialects()`
method and declare this list of Dialects explicitly.

### Initialization

In certain situations, a Pass may contain state that is constructed dynamically,
but is potentially expensive to recompute in successive runs of the Pass. One
such example is when using [`PDL`-based](Dialects/PDLOps.md)
[patterns](PatternRewriter.md), which are compiled into a bytecode during
runtime. In these situations, a pass may override the following hook to
initialize this heavy state:

*   `void initialize(MLIRContext *context)`

This hook is executed once per run of a full pass pipeline, meaning that it does
not have access to the state available during a `runOnOperation` call. More
concretely, all necessary accesses to an `MLIRContext` should be driven via the
provided `context` parameter, and methods that utilize "per-run" state such as
`getContext`/`getOperation`/`getAnalysis`/etc. must not be used.

## Analysis Management

An important concept, along with transformation passes, are analyses. These are
conceptually similar to transformation passes, except that they compute
information on a specific operation without modifying it. In MLIR, analyses are
not passes but free-standing classes that are computed lazily on-demand and
cached to avoid unnecessary recomputation. An analysis in MLIR must adhere to
the following:

*   Provide a valid constructor taking an `Operation*`.
*   Must not modify the given operation.

An analysis may provide additional hooks to control various behavior:

*   `bool isInvalidated(const AnalysisManager::PreservedAnalyses &)`

Given a preserved analysis set, the analysis returns true if it should truly be
invalidated. This allows for more fine-tuned invalidation in cases where an
analysis wasn't explicitly marked preserved, but may be preserved (or
invalidated) based upon other properties such as analyses sets.

### Querying Analyses

The base `OperationPass` class provides utilities for querying and preserving
analyses for the current operation being processed.

*   OperationPass automatically provides the following utilities for querying
    analyses:
    *   `getAnalysis<>`
        -   Get an analysis for the current operation, constructing it if
            necessary.
    *   `getCachedAnalysis<>`
        -   Get an analysis for the current operation, if it already exists.
    *   `getCachedParentAnalysis<>`
        -   Get an analysis for a given parent operation, if it exists.
    *   `getCachedChildAnalysis<>`
        -   Get an analysis for a given child operation, if it exists.
    *   `getChildAnalysis<>`
        -   Get an analysis for a given child operation, constructing it if
            necessary.

Using the example passes defined above, let's see some examples:

```c++
/// An interesting analysis.
struct MyOperationAnalysis {
  // Compute this analysis with the provided operation.
  MyOperationAnalysis(Operation *op);
};

void MyOperationPass::runOnOperation() {
  // Query MyOperationAnalysis for the current operation.
  MyOperationAnalysis &myAnalysis = getAnalysis<MyOperationAnalysis>();

  // Query a cached instance of MyOperationAnalysis for the current operation.
  // It will not be computed if it doesn't exist.
  auto optionalAnalysis = getCachedAnalysis<MyOperationAnalysis>();
  if (optionalAnalysis)
    ...

  // Query a cached instance of MyOperationAnalysis for the parent operation of
  // the current operation. It will not be computed if it doesn't exist.
  auto optionalAnalysis = getCachedParentAnalysis<MyOperationAnalysis>();
  if (optionalAnalysis)
    ...
}
```

### Preserving Analyses

Analyses that are constructed after being queried by a pass are cached to avoid
unnecessary computation if they are requested again later. To avoid stale
analyses, all analyses are assumed to be invalidated by a pass. To avoid
invalidation, a pass must specifically mark analyses that are known to be
preserved.

*   All Pass classes automatically provide the following utilities for
    preserving analyses:
    *   `markAllAnalysesPreserved`
    *   `markAnalysesPreserved<>`

```c++
void MyOperationPass::runOnOperation() {
  // Mark all analyses as preserved. This is useful if a pass can guarantee
  // that no transformation was performed.
  markAllAnalysesPreserved();

  // Mark specific analyses as preserved. This is used if some transformation
  // was performed, but some analyses were either unaffected or explicitly
  // preserved.
  markAnalysesPreserved<MyAnalysis, MyAnalyses...>();
}
```

## Pass Failure

Passes in MLIR are allowed to gracefully fail. This may happen if some invariant
of the pass was broken, potentially leaving the IR in some invalid state. If
such a situation occurs, the pass can directly signal a failure to the pass
manager via the `signalPassFailure` method. If a pass signaled a failure when
executing, no other passes in the pipeline will execute and the top-level call
to `PassManager::run` will return `failure`.

```c++
void MyOperationPass::runOnOperation() {
  // Signal failure on a broken invariant.
  if (some_broken_invariant)
    return signalPassFailure();
}
```

## Pass Manager

The above sections introduced the different types of passes and their
invariants. This section introduces the concept of a PassManager, and how it can
be used to configure and schedule a pass pipeline. There are two main classes
related to pass management, the `PassManager` and the `OpPassManager`. The
`PassManager` class acts as the top-level entry point, and contains various
configurations used for the entire pass pipeline. The `OpPassManager` class is
used to schedule passes to run at a specific level of nesting. The top-level
`PassManager` also functions as an `OpPassManager`.

### OpPassManager

An `OpPassManager` is essentially a collection of passes to execute on an
operation of a specific type. This operation type must adhere to the following
requirement:

*   Must be registered and marked
    [`IsolatedFromAbove`](Traits.md#isolatedfromabove).

    *   Passes are expected to not modify operations at or above the current
        operation being processed. If the operation is not isolated, it may
        inadvertently modify or traverse the SSA use-list of an operation it is
        not supposed to.

Passes can be added to a pass manager via `addPass`. The pass must either be an
`op-specific` pass operating on the same operation type as `OpPassManager`, or
an `op-agnostic` pass.

An `OpPassManager` is generally created by explicitly nesting a pipeline within
another existing `OpPassManager` via the `nest<>` method. This method takes the
operation type that the nested pass manager will operate on. At the top-level, a
`PassManager` acts as an `OpPassManager`. Nesting in this sense, corresponds to
the [structural](Tutorials/UnderstandingTheIRStructure.md) nesting within
[Regions](LangRef.md#regions) of the IR.

For example, the following `.mlir`:

```
module {
  spv.module "Logical" "GLSL450" {
    func @foo() {
      ...
    }
  }
}
```

Has the nesting structure of:

```
`module`
  `spv.module`
    `function`
```

Below is an example of constructing a pipeline that operates on the above
structure:

```c++
// Create a top-level `PassManager` class. If an operation type is not
// explicitly specific, the default is the builtin `module` operation.
PassManager pm(ctx);
// Note: We could also create the above `PassManager` this way.
PassManager pm(ctx, /*operationName=*/"module");

// Add a pass on the top-level module operation.
pm.addPass(std::make_unique<MyModulePass>());

// Nest a pass manager that operates on `spirv.module` operations nested
// directly under the top-level module.
OpPassManager &nestedModulePM = pm.nest<spirv::ModuleOp>();
nestedModulePM.addPass(std::make_unique<MySPIRVModulePass>());

// Nest a pass manager that operates on functions within the nested SPIRV
// module.
OpPassManager &nestedFunctionPM = nestedModulePM.nest<FuncOp>();
nestedFunctionPM.addPass(std::make_unique<MyFunctionPass>());

// Run the pass manager on the top-level module.
ModuleOp m = ...;
if (failed(pm.run(m)))
    ... // One of the passes signaled a failure.
```

The above pass manager contains the following pipeline structure:

```
OpPassManager<ModuleOp>
  MyModulePass
  OpPassManager<spirv::ModuleOp>
    MySPIRVModulePass
    OpPassManager<FuncOp>
      MyFunctionPass
```

These pipelines are then run over a single operation at a time. This means that,
for example, given a series of consecutive passes on FuncOp, it will execute all
on the first function, then all on the second function, etc. until the entire
program has been run through the passes. This provides several benefits:

*   This improves the cache behavior of the compiler, because it is only
    touching a single function at a time, instead of traversing the entire
    program.
*   This improves multi-threading performance by reducing the number of jobs
    that need to be scheduled, as well as increasing the efficiency of each job.
    An entire function pipeline can be run on each function asynchronously.

## Dynamic Pass Pipelines

In some situations it may be useful to run a pass pipeline within another pass,
to allow configuring or filtering based on some invariants of the current
operation being operated on. For example, the
[Inliner Pass](Passes.md#-inline-inline-function-calls) may want to run
intraprocedural simplification passes while it is inlining to produce a better
cost model, and provide more optimal inlining. To enable this, passes may run an
arbitrary `OpPassManager` on the current operation being operated on or any
operation nested within the current operation via the `LogicalResult
Pass::runPipeline(OpPassManager &, Operation *)` method. This method returns
whether the dynamic pipeline succeeded or failed, similarly to the result of the
top-level `PassManager::run` method. A simple example is shown below:

```c++
void MyModulePass::runOnOperation() {
  ModuleOp module = getOperation();
  if (hasSomeSpecificProperty(module)) {
    OpPassManager dynamicPM("module");
    ...; // Build the dynamic pipeline.
    if (failed(runPipeline(dynamicPM, module)))
      return signalPassFailure();
  }
}
```

Note: though above the dynamic pipeline was constructed within the
`runOnOperation` method, this is not necessary and pipelines should be cached
when possible as the `OpPassManager` class can be safely copy constructed.

The mechanism described in this section should be used whenever a pass pipeline
should run in a nested fashion, i.e. when the nested pipeline cannot be
scheduled statically along with the rest of the main pass pipeline. More
specifically, a `PassManager` should generally never need to be constructed
within a `Pass`. Using `runPipeline` also ensures that all analyses,
[instrumentations](#pass-instrumentation), and other pass manager related
components are integrated with the dynamic pipeline being executed.

## Instance Specific Pass Options

MLIR provides a builtin mechanism for passes to specify options that configure
its behavior. These options are parsed at pass construction time independently
for each instance of the pass. Options are defined using the `Option<>` and
`ListOption<>` classes, and follow the
[LLVM command line](https://llvm.org/docs/CommandLine.html) flag definition
rules. See below for a few examples:

```c++
struct MyPass ... {
  /// Make sure that we have a valid default constructor and copy constructor to
  /// ensure that the options are initialized properly.
  MyPass() = default;
  MyPass(const MyPass& pass) {}

  /// Any parameters after the description are forwarded to llvm::cl::list and
  /// llvm::cl::opt respectively.
  Option<int> exampleOption{*this, "flag-name", llvm::cl::desc("...")};
  ListOption<int> exampleListOption{*this, "list-flag-name",
                                    llvm::cl::desc("...")};
};
```

For pass pipelines, the `PassPipelineRegistration` templates take an additional
template parameter for an optional `Option` struct definition. This struct
should inherit from `mlir::PassPipelineOptions` and contain the desired pipeline
options. When using `PassPipelineRegistration`, the constructor now takes a
function with the signature `void (OpPassManager &pm, const MyPipelineOptions&)`
which should construct the passes from the options and pass them to the pm:

```c++
struct MyPipelineOptions : public PassPipelineOptions {
  // The structure of these options is the same as those for pass options.
  Option<int> exampleOption{*this, "flag-name", llvm::cl::desc("...")};
  ListOption<int> exampleListOption{*this, "list-flag-name",
                                    llvm::cl::desc("...")};
};

void registerMyPasses() {
  PassPipelineRegistration<MyPipelineOptions>(
    "example-pipeline", "Run an example pipeline.",
    [](OpPassManager &pm, const MyPipelineOptions &pipelineOptions) {
      // Initialize the pass manager.
    });
}
```

## Pass Statistics

Statistics are a way to keep track of what the compiler is doing and how
effective various transformations are. It is often useful to see what effect
specific transformations have on a particular input, and how often they trigger.
Pass statistics are specific to each pass instance, which allow for seeing the
effect of placing a particular transformation at specific places within the pass
pipeline. For example, they help answer questions like "What happens if I run
CSE again here?".

Statistics can be added to a pass by using the 'Pass::Statistic' class. This
class takes as a constructor arguments: the parent pass, a name, and a
description. This class acts like an atomic unsigned integer, and may be
incremented and updated accordingly. These statistics rely on the same
infrastructure as
[`llvm::Statistic`](http://llvm.org/docs/ProgrammersManual.html#the-statistic-class-stats-option)
and thus have similar usage constraints. Collected statistics can be dumped by
the [pass manager](#pass-manager) programmatically via
`PassManager::enableStatistics`; or via `-pass-statistics` and
`-pass-statistics-display` on the command line.

An example is shown below:

```c++
struct MyPass ... {
  /// Make sure that we have a valid default constructor and copy constructor to
  /// ensure that the options are initialized properly.
  MyPass() = default;
  MyPass(const MyPass& pass) {}

  /// Define the statistic to track during the execution of MyPass.
  Statistic exampleStat{this, "exampleStat", "An example statistic"};

  void runOnOperation() {
    ...

    // Update the statistic after some invariant was hit.
    ++exampleStat;

    ...
  }
};
```

The collected statistics may be aggregated in two types of views:

A pipeline view that models the structure of the pass manager, this is the
default view:

```shell
$ mlir-opt -pass-pipeline='func(my-pass,my-pass)' foo.mlir -pass-statistics

===-------------------------------------------------------------------------===
                         ... Pass statistics report ...
===-------------------------------------------------------------------------===
'func' Pipeline
  MyPass
    (S) 15 exampleStat - An example statistic
  VerifierPass
  MyPass
    (S)  6 exampleStat - An example statistic
  VerifierPass
VerifierPass
```

A list view that aggregates the statistics of all instances of a specific pass
together:

```shell
$ mlir-opt -pass-pipeline='func(my-pass, my-pass)' foo.mlir -pass-statistics -pass-statistics-display=list

===-------------------------------------------------------------------------===
                         ... Pass statistics report ...
===-------------------------------------------------------------------------===
MyPass
  (S) 21 exampleStat - An example statistic
```

## Pass Registration

Briefly shown in the example definitions of the various pass types is the
`PassRegistration` class. This mechanism allows for registering pass classes so
that they may be created within a
[textual pass pipeline description](#textual-pass-pipeline-specification). An
example registration is shown below:

```c++
void registerMyPass() {
  PassRegistration<MyPass>("argument", "description");
}
```

*   `MyPass` is the name of the derived pass class.
*   "argument" is the argument used to refer to the pass in the textual format.
*   "description" is a brief description of the pass.

For passes that cannot be default-constructed, `PassRegistration` accepts an
optional third argument that takes a callback to create the pass:

```c++
void registerMyPass() {
  PassRegistration<MyParametricPass>(
    "argument", "description",
    []() -> std::unique_ptr<Pass> {
      std::unique_ptr<Pass> p = std::make_unique<MyParametricPass>(/*options*/);
      /*... non-trivial-logic to configure the pass ...*/;
      return p;
    });
}
```

This variant of registration can be used, for example, to accept the
configuration of a pass from command-line arguments and pass it to the pass
constructor.

Note: Make sure that the pass is copy-constructible in a way that does not share
data as the [pass manager](#pass-manager) may create copies of the pass to run
in parallel.

### Pass Pipeline Registration

Described above is the mechanism used for registering a specific derived pass
class. On top of that, MLIR allows for registering custom pass pipelines in a
similar fashion. This allows for custom pipelines to be available to tools like
mlir-opt in the same way that passes are, which is useful for encapsulating
common pipelines like the "-O1" series of passes. Pipelines are registered via a
similar mechanism to passes in the form of `PassPipelineRegistration`. Compared
to `PassRegistration`, this class takes an additional parameter in the form of a
pipeline builder that modifies a provided `OpPassManager`.

```c++
void pipelineBuilder(OpPassManager &pm) {
  pm.addPass(std::make_unique<MyPass>());
  pm.addPass(std::make_unique<MyOtherPass>());
}

void registerMyPasses() {
  // Register an existing pipeline builder function.
  PassPipelineRegistration<>(
    "argument", "description", pipelineBuilder);

  // Register an inline pipeline builder.
  PassPipelineRegistration<>(
    "argument", "description", [](OpPassManager &pm) {
      pm.addPass(std::make_unique<MyPass>());
      pm.addPass(std::make_unique<MyOtherPass>());
    });
}
```

### Textual Pass Pipeline Specification

The previous sections detailed how to register passes and pass pipelines with a
specific argument and description. Once registered, these can be used to
configure a pass manager from a string description. This is especially useful
for tools like `mlir-opt`, that configure pass managers from the command line,
or as options to passes that utilize
[dynamic pass pipelines](#dynamic-pass-pipelines).

To support the ability to describe the full structure of pass pipelines, MLIR
supports a custom textual description of pass pipelines. The textual description
includes the nesting structure, the arguments of the passes and pass pipelines
to run, and any options for those passes and pipelines. A textual pipeline is
defined as a series of names, each of which may in itself recursively contain a
nested pipeline description. The syntax for this specification is as follows:

```ebnf
pipeline          ::= op-name `(` pipeline-element (`,` pipeline-element)* `)`
pipeline-element  ::= pipeline | (pass-name | pass-pipeline-name) options?
options           ::= '{' (key ('=' value)?)+ '}'
```

*   `op-name`
    *   This corresponds to the mnemonic name of an operation to run passes on,
        e.g. `func` or `module`.
*   `pass-name` | `pass-pipeline-name`
    *   This corresponds to the argument of a registered pass or pass pipeline,
        e.g. `cse` or `canonicalize`.
*   `options`
    *   Options are specific key value pairs representing options defined by a
        pass or pass pipeline, as described in the
        ["Instance Specific Pass Options"](#instance-specific-pass-options)
        section. See this section for an example usage in a textual pipeline.

For example, the following pipeline:

```shell
$ mlir-opt foo.mlir -cse -canonicalize -convert-std-to-llvm='use-bare-ptr-memref-call-conv=1'
```

Can also be specified as (via the `-pass-pipeline` flag):

```shell
$ mlir-opt foo.mlir -pass-pipeline='func(cse,canonicalize),convert-std-to-llvm{use-bare-ptr-memref-call-conv=1}'
```

In order to support round-tripping a pass to the textual representation using
`OpPassManager::printAsTextualPipeline(raw_ostream&)`, override `StringRef
Pass::getArgument()` to specify the argument used when registering a pass.

## Declarative Pass Specification

Some aspects of a Pass may be specified declaratively, in a form similar to
[operations](OpDefinitions.md). This specification simplifies several mechanisms
used when defining passes. It can be used for generating pass registration
calls, defining boilerplate pass utilities, and generating pass documentation.

Consider the following pass specified in C++:

```c++
struct MyPass : PassWrapper<MyPass, OperationPass<ModuleOp>> {
  MyPass() = default;
  MyPass(const MyPass &) {}

  ...

  // Specify any options.
  Option<bool> option{
      *this, "example-option",
      llvm::cl::desc("An example option"), llvm::cl::init(true)};
  ListOption<int64_t> listOption{
      *this, "example-list",
      llvm::cl::desc("An example list option"), llvm::cl::ZeroOrMore,
      llvm::cl::MiscFlags::CommaSeparated};

  // Specify any statistics.
  Statistic statistic{this, "example-statistic", "An example statistic"};
};

/// Expose this pass to the outside world.
std::unique_ptr<Pass> foo::createMyPass() {
  return std::make_unique<MyPass>();
}

/// Register this pass.
void foo::registerMyPass() {
  PassRegistration<MyPass>("my-pass", "My pass summary");
}
```

This pass may be specified declaratively as so:

```tablegen
def MyPass : Pass<"my-pass", "ModuleOp"> {
  let summary = "My Pass Summary";
  let description = [{
    Here we can now give a much larger description of `MyPass`, including all of
    its various constraints and behavior.
  }];

  // A constructor must be provided to specify how to create a default instance
  // of MyPass.
  let constructor = "foo::createMyPass()";

  // Specify any options.
  let options = [
    Option<"option", "example-option", "bool", /*default=*/"true",
           "An example option">,
    ListOption<"listOption", "example-list", "int64_t",
               "An example list option",
               "llvm::cl::ZeroOrMore, llvm::cl::MiscFlags::CommaSeparated">
  ];

  // Specify any statistics.
  let statistics = [
    Statistic<"statistic", "example-statistic", "An example statistic">
  ];
}
```

Using the `gen-pass-decls` generator, we can generate most of the boilerplate
above automatically. This generator takes as an input a `-name` parameter, that
provides a tag for the group of passes that are being generated. This generator
produces two chunks of output:

The first is a code block for registering the declarative passes with the global
registry. For each pass, the generator produces a `registerFooPass` where `Foo`
is the name of the definition specified in tablegen. It also generates a
`registerGroupPasses`, where `Group` is the tag provided via the `-name` input
parameter, that registers all of the passes present.

```c++
// gen-pass-decls -name="Example"

#define GEN_PASS_REGISTRATION
#include "Passes.h.inc"

void registerMyPasses() {
  // Register all of the passes.
  registerExamplePasses();

  // Register `MyPass` specifically.
  registerMyPassPass();
}
```

The second is a base class for each of the passes, containing most of the boiler
plate related to pass definitions. These classes are named in the form of
`MyPassBase`, where `MyPass` is the name of the pass definition in tablegen. We
can update the original C++ pass definition as so:

```c++
/// Include the generated base pass class definitions.
#define GEN_PASS_CLASSES
#include "Passes.h.inc"

/// Define the main class as deriving from the generated base class.
struct MyPass : MyPassBase<MyPass> {
  /// The explicit constructor is no longer explicitly necessary when defining
  /// pass options and statistics, the base class takes care of that
  /// automatically.
  ...

  /// The definitions of the options and statistics are now generated within
  /// the base class, but are accessible in the same way.
};

/// Expose this pass to the outside world.
std::unique_ptr<Pass> foo::createMyPass() {
  return std::make_unique<MyPass>();
}
```

Using the `gen-pass-doc` generator, markdown documentation for each of the
passes can be generated. See [Passes.md](Passes.md) for example output of real
MLIR passes.

### Tablegen Specification

The `Pass` class is used to begin a new pass definition. This class takes as an
argument the registry argument to attribute to the pass, as well as an optional
string corresponding to the operation type that the pass operates on. The class
contains the following fields:

*   `summary`
    -   A short one line summary of the pass, used as the description when
        registering the pass.
*   `description`
    -   A longer, more detailed description of the pass. This is used when
        generating pass documentation.
*   `dependentDialects`
    -   A list of strings representing the `Dialect` classes this pass may
        introduce entities, Attributes/Operations/Types/etc., of.
*   `constructor`
    -   A code block used to create a default instance of the pass.
*   `options`
    -   A list of pass options used by the pass.
*   `statistics`
    -   A list of pass statistics used by the pass.

#### Options

Options may be specified via the `Option` and `ListOption` classes. The `Option`
class takes the following template parameters:

*   C++ variable name
    -   A name to use for the generated option variable.
*   argument
    -   The argument name of the option.
*   type
    -   The C++ type of the option.
*   default value
    -   The default option value.
*   description
    -   A one line description of the option.
*   additional option flags
    -   A string containing any additional options necessary to construct the
        option.

```tablegen
def MyPass : Pass<"my-pass"> {
  let options = [
    Option<"option", "example-option", "bool", /*default=*/"true",
           "An example option">,
  ];
}
```

The `ListOption` class takes the following fields:

*   C++ variable name
    -   A name to use for the generated option variable.
*   argument
    -   The argument name of the option.
*   element type
    -   The C++ type of the list element.
*   description
    -   A one line description of the option.
*   additional option flags
    -   A string containing any additional options necessary to construct the
        option.

```tablegen
def MyPass : Pass<"my-pass"> {
  let options = [
    ListOption<"listOption", "example-list", "int64_t",
               "An example list option",
               "llvm::cl::ZeroOrMore, llvm::cl::MiscFlags::CommaSeparated">
  ];
}
```

#### Statistic

Statistics may be specified via the `Statistic`, which takes the following
template parameters:

*   C++ variable name
    -   A name to use for the generated statistic variable.
*   display name
    -   The name used when displaying the statistic.
*   description
    -   A one line description of the statistic.

```tablegen
def MyPass : Pass<"my-pass"> {
  let statistics = [
    Statistic<"statistic", "example-statistic", "An example statistic">
  ];
}
```

## Pass Instrumentation

MLIR provides a customizable framework to instrument pass execution and analysis
computation, via the `PassInstrumentation` class. This class provides hooks into
the PassManager that observe various events:

*   `runBeforePipeline`
    *   This callback is run just before a pass pipeline, i.e. pass manager, is
        executed.
*   `runAfterPipeline`
    *   This callback is run right after a pass pipeline has been executed,
        successfully or not.
*   `runBeforePass`
    *   This callback is run just before a pass is executed.
*   `runAfterPass`
    *   This callback is run right after a pass has been successfully executed.
        If this hook is executed, `runAfterPassFailed` will *not* be.
*   `runAfterPassFailed`
    *   This callback is run right after a pass execution fails. If this hook is
        executed, `runAfterPass` will *not* be.
*   `runBeforeAnalysis`
    *   This callback is run just before an analysis is computed.
*   `runAfterAnalysis`
    *   This callback is run right after an analysis is computed.

PassInstrumentation instances may be registered directly with a
[PassManager](#pass-manager) instance via the `addInstrumentation` method.
Instrumentations added to the PassManager are run in a stack like fashion, i.e.
the last instrumentation to execute a `runBefore*` hook will be the first to
execute the respective `runAfter*` hook. The hooks of a `PassInstrumentation`
class are guaranteed to be executed in a thread safe fashion, so additional
synchronization is not necessary. Below in an example instrumentation that
counts the number of times the `DominanceInfo` analysis is computed:

```c++
struct DominanceCounterInstrumentation : public PassInstrumentation {
  /// The cumulative count of how many times dominance has been calculated.
  unsigned &count;

  DominanceCounterInstrumentation(unsigned &count) : count(count) {}
  void runAfterAnalysis(llvm::StringRef, TypeID id, Operation *) override {
    if (id == TypeID::get<DominanceInfo>())
      ++count;
  }
};

MLIRContext *ctx = ...;
PassManager pm(ctx);

// Add the instrumentation to the pass manager.
unsigned domInfoCount;
pm.addInstrumentation(
    std::make_unique<DominanceCounterInstrumentation>(domInfoCount));

// Run the pass manager on a module operation.
ModuleOp m = ...;
if (failed(pm.run(m)))
    ...

llvm::errs() << "DominanceInfo was computed " << domInfoCount << " times!\n";
```

### Standard Instrumentations

MLIR utilizes the pass instrumentation framework to provide a few useful
developer tools and utilities. Each of these instrumentations are directly
available to all users of the MLIR pass framework.

#### Pass Timing

The PassTiming instrumentation provides timing information about the execution
of passes and computation of analyses. This provides a quick glimpse into what
passes are taking the most time to execute, as well as how much of an effect a
pass has on the total execution time of the pipeline. Users can enable this
instrumentation directly on the PassManager via `enableTiming`. This
instrumentation is also made available in mlir-opt via the `-pass-timing` flag.
The PassTiming instrumentation provides several different display modes for the
timing results, each of which is described below:

##### List Display Mode

In this mode, the results are displayed in a list sorted by total time with each
pass/analysis instance aggregated into one unique result. This view is useful
for getting an overview of what analyses/passes are taking the most time in a
pipeline. This display mode is available in mlir-opt via
`-pass-timing-display=list`.

```shell
$ mlir-opt foo.mlir -mlir-disable-threading -pass-pipeline='func(cse,canonicalize)' -convert-std-to-llvm -pass-timing -pass-timing-display=list

===-------------------------------------------------------------------------===
                      ... Pass execution timing report ...
===-------------------------------------------------------------------------===
  Total Execution Time: 0.0203 seconds

   ---Wall Time---  --- Name ---
   0.0047 ( 55.9%)  Canonicalizer
   0.0019 ( 22.2%)  VerifierPass
   0.0016 ( 18.5%)  LLVMLoweringPass
   0.0003 (  3.4%)  CSE
   0.0002 (  1.9%)  (A) DominanceInfo
   0.0084 (100.0%)  Total
```

##### Pipeline Display Mode

In this mode, the results are displayed in a nested pipeline view that mirrors
the internal pass pipeline that is being executed in the pass manager. This view
is useful for understanding specifically which parts of the pipeline are taking
the most time, and can also be used to identify when analyses are being
invalidated and recomputed. This is the default display mode.

```shell
$ mlir-opt foo.mlir -mlir-disable-threading -pass-pipeline='func(cse,canonicalize)' -convert-std-to-llvm -pass-timing

===-------------------------------------------------------------------------===
                      ... Pass execution timing report ...
===-------------------------------------------------------------------------===
  Total Execution Time: 0.0249 seconds

   ---Wall Time---  --- Name ---
   0.0058 ( 70.8%)  'func' Pipeline
   0.0004 (  4.3%)    CSE
   0.0002 (  2.6%)      (A) DominanceInfo
   0.0004 (  4.8%)    VerifierPass
   0.0046 ( 55.4%)    Canonicalizer
   0.0005 (  6.2%)    VerifierPass
   0.0005 (  5.8%)  VerifierPass
   0.0014 ( 17.2%)  LLVMLoweringPass
   0.0005 (  6.2%)  VerifierPass
   0.0082 (100.0%)  Total
```

##### Multi-threaded Pass Timing

When multi-threading is enabled in the pass manager the meaning of the display
slightly changes. First, a new timing column is added, `User Time`, that
displays the total time spent across all threads. Secondly, the `Wall Time`
column displays the longest individual time spent amongst all of the threads.
This means that the `Wall Time` column will continue to give an indicator on the
perceived time, or clock time, whereas the `User Time` will display the total
cpu time.

```shell
$ mlir-opt foo.mlir -pass-pipeline='func(cse,canonicalize)' -convert-std-to-llvm -pass-timing

===-------------------------------------------------------------------------===
                      ... Pass execution timing report ...
===-------------------------------------------------------------------------===
  Total Execution Time: 0.0078 seconds

   ---User Time---   ---Wall Time---  --- Name ---
   0.0177 ( 88.5%)     0.0057 ( 71.3%)  'func' Pipeline
   0.0044 ( 22.0%)     0.0015 ( 18.9%)    CSE
   0.0029 ( 14.5%)     0.0012 ( 15.2%)      (A) DominanceInfo
   0.0038 ( 18.9%)     0.0015 ( 18.7%)    VerifierPass
   0.0089 ( 44.6%)     0.0025 ( 31.1%)    Canonicalizer
   0.0006 (  3.0%)     0.0002 (  2.6%)    VerifierPass
   0.0004 (  2.2%)     0.0004 (  5.4%)  VerifierPass
   0.0013 (  6.5%)     0.0013 ( 16.3%)  LLVMLoweringPass
   0.0006 (  2.8%)     0.0006 (  7.0%)  VerifierPass
   0.0200 (100.0%)     0.0081 (100.0%)  Total
```

#### IR Printing

When debugging it is often useful to dump the IR at various stages of a pass
pipeline. This is where the IR printing instrumentation comes into play. This
instrumentation allows for conditionally printing the IR before and after pass
execution by optionally filtering on the pass being executed. This
instrumentation can be added directly to the PassManager via the
`enableIRPrinting` method. `mlir-opt` provides a few useful flags for utilizing
this instrumentation:

*   `print-ir-before=(comma-separated-pass-list)`
    *   Print the IR before each of the passes provided within the pass list.
*   `print-ir-before-all`
    *   Print the IR before every pass in the pipeline.

```shell
$ mlir-opt foo.mlir -pass-pipeline='func(cse)' -print-ir-before=cse

*** IR Dump Before CSE ***
func @simple_constant() -> (i32, i32) {
  %c1_i32 = constant 1 : i32
  %c1_i32_0 = constant 1 : i32
  return %c1_i32, %c1_i32_0 : i32, i32
}
```

*   `print-ir-after=(comma-separated-pass-list)`
    *   Print the IR after each of the passes provided within the pass list.
*   `print-ir-after-all`
    *   Print the IR after every pass in the pipeline.

```shell
$ mlir-opt foo.mlir -pass-pipeline='func(cse)' -print-ir-after=cse

*** IR Dump After CSE ***
func @simple_constant() -> (i32, i32) {
  %c1_i32 = constant 1 : i32
  return %c1_i32, %c1_i32 : i32, i32
}
```

*   `print-ir-after-change`
    *   Only print the IR after a pass if the pass mutated the IR. This helps to
        reduce the number of IR dumps for "uninteresting" passes.
    *   Note: Changes are detected by comparing a hash of the operation before
        and after the pass. This adds additional run-time to compute the hash of
        the IR, and in some rare cases may result in false-positives depending
        on the collision rate of the hash algorithm used.
    *   Note: This option should be used in unison with one of the other
        'print-ir-after' options above, as this option alone does not enable
        printing.

```shell
$ mlir-opt foo.mlir -pass-pipeline='func(cse,cse)' -print-ir-after=cse -print-ir-after-change

*** IR Dump After CSE ***
func @simple_constant() -> (i32, i32) {
  %c1_i32 = constant 1 : i32
  return %c1_i32, %c1_i32 : i32, i32
}
```

*   `print-ir-module-scope`
    *   Always print the top-level module operation, regardless of pass type or
        operation nesting level.
    *   Note: Printing at module scope should only be used when multi-threading
        is disabled(`-mlir-disable-threading`)

```shell
$ mlir-opt foo.mlir -mlir-disable-threading -pass-pipeline='func(cse)' -print-ir-after=cse -print-ir-module-scope

*** IR Dump After CSE ***  ('func' operation: @bar)
func @bar(%arg0: f32, %arg1: f32) -> f32 {
  ...
}

func @simple_constant() -> (i32, i32) {
  %c1_i32 = constant 1 : i32
  %c1_i32_0 = constant 1 : i32
  return %c1_i32, %c1_i32_0 : i32, i32
}

*** IR Dump After CSE ***  ('func' operation: @simple_constant)
func @bar(%arg0: f32, %arg1: f32) -> f32 {
  ...
}

func @simple_constant() -> (i32, i32) {
  %c1_i32 = constant 1 : i32
  return %c1_i32, %c1_i32 : i32, i32
}
```

## Crash and Failure Reproduction

The [pass manager](#pass-manager) in MLIR contains a builtin mechanism to
generate reproducibles in the even of a crash, or a
[pass failure](#pass-failure). This functionality can be enabled via
`PassManager::enableCrashReproducerGeneration` or via the command line flag
`pass-pipeline-crash-reproducer`. In either case, an argument is provided that
corresponds to the output `.mlir` file name that the reproducible should be
written to. The reproducible contains the configuration of the pass manager that
was executing, as well as the initial IR before any passes were run. A potential
reproducible may have the form:

```mlir
// configuration: -pass-pipeline='func(cse,canonicalize),inline' -verify-each

module {
  func @foo() {
    ...
  }
}
```

The configuration dumped can be passed to `mlir-opt` by specifying
`-run-reproducer` flag. This will result in parsing the first line configuration
of the reproducer and adding those to the command line options.

Beyond specifying a filename, one can also register a `ReproducerStreamFactory`
function that would be invoked in the case of a crash and the reproducer written
to its stream.

### Local Reproducer Generation

An additional flag may be passed to
`PassManager::enableCrashReproducerGeneration`, and specified via
`pass-pipeline-local-reproducer` on the command line, that signals that the pass
manager should attempt to generate a "local" reproducer. This will attempt to
generate a reproducer containing IR right before the pass that fails. This is
useful for situations where the crash is known to be within a specific pass, or
when the original input relies on components (like dialects or passes) that may
not always be available.

For example, if the failure in the previous example came from `canonicalize`,
the following reproducer will be generated:

```mlir
// configuration: -pass-pipeline='func(canonicalize)'
// note: verifyPasses=false

module {
  func @foo() {
    ...
  }
}
```
