#--- podExamples.t ------------------------------------------------------------
# function: Test HTML::ToC.  In particular test the examples as described in 
#           the POD documentation.

use strict;
use Test;

BEGIN { plan tests => 13; }

use HTML::Toc;
use HTML::TocGenerator;
use HTML::TocInsertor;
use HTML::TocUpdator;

my ($filename, $filename2);
	

BEGIN {
		# Create test file
	$filename = "tmp.htm";
	die "$filename is already there" if -e $filename;
		# Create test file 2
	$filename2 = "tmp2.htm";
	die "$filename2 is already there" if -e $filename2;
}


END {
		# Remove test file
	unlink($filename) or warn "Can't unlink $filename: $!";
		# Remove test file 2
	unlink($filename2) or warn "Can't unlink $filename2: $!";
}


#--- TestExtendFromFile() --------------------------------------------------
# function: Test 'HTML::TocGenerator->extendFromFile()

sub TestExtendFromFile {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter of document 1</h1>
</body>
EOT

		# Assemble test file 2
	open(FILE, ">$filename2") || die "Can't create $filename2: $!";
	print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>Chapter of document 2</h1>
</body>
</html>
EOT

		# Create objects
	my $toc          = HTML::Toc->new();
	my $tocGenerator = HTML::TocGenerator->new();
	
		# Set ToC options
   $toc->setOptions({'doLinkToFile' => 1});
		# Generate ToC
	$tocGenerator->generateFromFile($toc, $filename);
	$tocGenerator->extendFromFile($toc, $filename2);
		# Test ToC
	ok($toc->format(), <<EOT);

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=tmp.htm#h-1>Chapter of document 1</a>
   <li><a href=tmp2.htm#h-2>Chapter of document 2</a>
</ul>
<!-- End of generated Table of Contents -->
EOT
}  # TestExtendFromFile()


#--- TestGenerateFromFiles() --------------------------------------------------
# function: Test 'HTML::TocGenerator->generateFromFile()

sub TestGenerateFromFiles {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter of document 1</h1>
</body>
EOT

		# Assemble test file 2
	open(FILE, ">$filename2") || die "Can't create $filename2: $!";
	print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>Chapter of document 2</h1>
</body>
</html>
EOT

		# Create objects
	my $toc          = HTML::Toc->new();
	my $tocGenerator = HTML::TocGenerator->new();
	
		# Set ToC options
   $toc->setOptions({'doLinkToFile' => 1});
		# Generate ToC
	$tocGenerator->generateFromFile($toc, [$filename, $filename2]);
		# Test ToC
	ok($toc->format(), <<EOT);

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=tmp.htm#h-1>Chapter of document 1</a>
   <li><a href=tmp2.htm#h-2>Chapter of document 2</a>
</ul>
<!-- End of generated Table of Contents -->
EOT
}  # TestGenerateFromFiles()


#--- TestGenerateFromFiles() --------------------------------------------------
# function: Test 'HTML::TocGenerator->generateFromFile() using multiple files.

sub TestGenerateFromFile {
		# Assemble test file 1
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>Chapter</h1>
</body>
</html>
EOT

		# Create objects
	my $toc          = HTML::Toc->new();
	my $tocGenerator = HTML::TocGenerator->new();
	
		# Generate ToC
	$tocGenerator->generateFromFile($toc, $filename);
		# Test ToC
	ok($toc->format(), <<EOT);

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#h-1>Chapter</a>
</ul>
<!-- End of generated Table of Contents -->
EOT
}  # TestGenerateFromFile()


#--- TestInsertIntoFile() -----------------------------------------------------
# function: Test 'HTML::TocInsertor->insertIntoFile()

sub TestInsertIntoFile {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>Chapter</h1>
</body>
</html>
EOT

		# Create objects
	my $toc         = HTML::Toc->new();
	my $tocInsertor = HTML::TocInsertor->new();
	my $output;
	
		# Generate ToC
	$tocInsertor->insertIntoFile($toc, $filename, {'output' => \$output});
		# Test ToC
	ok($output, <<EOT);
<html>
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#h-1>Chapter</a>
</ul>
<!-- End of generated Table of Contents -->

   <a name=h-1><h1>Chapter</h1></a>
</body>
</html>
EOT
}  # TestInsertIntoFile()


#--- TestInsertIntoFileUsingTocUpdator() --------------------------------------
# function: Test 'HTML::TocUpdator->insertIntoFile()

sub TestInsertIntoFileUsingTocUpdator {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>
   Chapter
   </h1>
</body>
</html>
EOT

		# Create objects
	my $toc        = HTML::Toc->new();
	my $tocUpdator = HTML::TocUpdator->new();
	my $output;
	
		# Generate ToC
	$tocUpdator->insertIntoFile($toc, $filename, {'output' => \$output});
		# Test ToC
	ok($output, <<EOT);
<html>
<body><!-- #BeginToc -->
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#h-1> Chapter </a>
</ul>
<!-- End of generated Table of Contents -->
<!-- #EndToc -->
   <!-- #BeginTocAnchorNameBegin --><a name=h-1><!-- #EndTocAnchorNameBegin --><h1>
   Chapter
   </h1><!-- #BeginTocAnchorNameEnd --></a><!-- #EndTocAnchorNameEnd -->
</body>
</html>
EOT
}  # TestInsertIntoFileUsingTocUpdator()


#--- TestGlobalGroups0() ------------------------------------------------------
# function: Test 'HTML::TocGenerator' option 'doUseGroupsGlobal = 0'.

sub TestGlobalGroups0 {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<h1>Chapter</h1>
<h2>Paragraph</h2>
EOT

		# Create objects
	my $toc1         = HTML::Toc->new();
	my $toc2         = HTML::Toc->new();
	my $tocGenerator = HTML::TocGenerator->new();

		# Set options
	$toc1->setOptions({
		'header'      => '',
		'footer'      => '',
		'tokenToToc' => [{'tokenBegin' => '<h1>'}]
	});
	$toc2->setOptions({
		'header'      => '',
		'footer'      => '',
		'tokenToToc' => [{'tokenBegin' => '<h2>'}]
	});
		# Generate ToC
	$tocGenerator->generateFromFile([$toc1, $toc2], $filename);
		# Test ToC
	ok($toc1->format() . $toc2->format() . "\n", <<'EOT');
<ul>
   <li><a href=#h-1>Chapter</a>
</ul><ul>
   <li><a href=#h-1>Paragraph</a>
</ul>
EOT
}	# TestGlobalGroups0()


#--- TestGlobalGroups1() ------------------------------------------------------
# function: Test 'HTML::TocGenerator' option 'doUseGroupsGlobal = 0'.

sub TestGlobalGroups1 {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT';
<h1>Chapter</h1>
<h2>Paragraph</h2>
EOT
	close(FILE);

		# Create objects
	my $toc1         = HTML::Toc->new();
	my $toc2         = HTML::Toc->new();
	my $tocGenerator = HTML::TocGenerator->new();

		# Set options
	$toc1->setOptions({
		'header'      => '',
		'footer'      => '',
		'tokenToToc' => [{'tokenBegin' => '<h1>'}]
	});
	$toc2->setOptions({
		'header'      => '',
		'footer'      => '',
		'tokenToToc' => [{'tokenBegin' => '<h2>'}]
	});
		# Generate ToC
	$tocGenerator->generateFromFile(
		[$toc1, $toc2], $filename, {'doUseGroupsGlobal' => 1}
	);
		# Test ToC
	ok($toc1->format() . $toc2->format() . "\n", <<'EOT');
<ul>
   <li><a href=#h-1>Chapter</a>
</ul><ul>
   <li><a href=#h-2>Paragraph</a>
</ul>
EOT
}	# TestGlobalGroups1()


#--- TestMultipleGroupsAppendix() ---------------------------------------------
# function: Test multiple ToCs

sub TestMultipleGroupsAppendix() {
		# Create objects
	my $toc         = HTML::Toc->new();
	my $tocInsertor = HTML::TocInsertor->new();
	my $output;
	
		# Set ToC options
	$toc->setOptions({
		'tokenToToc' => [{
				'tokenBegin' => '<h1 class=-appendix>'
			}, {
				'tokenBegin' => '<h2 class=-appendix>',
				'level'      => 2
			}, {
				'groupId'    => 'appendix',
				'tokenBegin' => '<h1 class=appendix>',
			}, {
				'groupId'    => 'appendix',
				'tokenBegin' => '<h2 class=appendix>',
				'level'      => 2
			}],
	});
		# Generate ToC
	$tocInsertor->insert($toc, <<EOT, {'output' => \$output});
<body>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
   <h3>Subparagraph</h3>
   <h1>Chapter</h1>
   <h1 class=appendix>Appendix Chapter</h1>
   <h2 class=appendix>Appendix Paragraph</h2>
</body>
EOT
		# Test ToC
	ok($output, <<EOT);
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#h-1>Chapter</a>
   <ul>
      <li><a href=#h-1.1>Paragraph</a>
   </ul>
   <li><a href=#h-2>Chapter</a>
</ul>
<ul>
   <li><a href=#appendix-1>Appendix Chapter</a>
   <ul>
      <li><a href=#appendix-1.1>Appendix Paragraph</a>
   </ul>
</ul>
<!-- End of generated Table of Contents -->

   <a name=h-1><h1>Chapter</h1></a>
   <a name=h-1.1><h2>Paragraph</h2></a>
   <h3>Subparagraph</h3>
   <a name=h-2><h1>Chapter</h1></a>
   <a name=appendix-1><h1 class=appendix>Appendix Chapter</h1></a>
   <a name=appendix-1.1><h2 class=appendix>Appendix Paragraph</h2></a>
</body>
EOT
}  # TestMultipleGroupsAppendix()


#--- TestMultipleGroupsPart() -------------------------------------------------
# function: Test multiple ToCs

sub TestMultipleGroupsPart() {
		# Create objects
	my $toc         = HTML::Toc->new();
	my $tocInsertor = HTML::TocInsertor->new();
	my $output;
	
		# Set ToC options
	$toc->setOptions({
		'tokenToToc' => [{
				'tokenBegin' => '<h1 class=-part>'
			}, {
				'tokenBegin' => '<h2 class=-part>',
				'level'      => 2,
			}, {
				'groupId'        => 'part',
				'tokenBegin'     => '<h1 class=part>',
				'level'          => 1,
				'doNumberToken'  => 1,
				'numberingStyle' => 'upper-alpha'
			}]
	});
		# Generate ToC
	$tocInsertor->insert($toc, <<EOT, {'output' => \$output});
<body>
   <h1 class=part>First Part</h1>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
   <h1 class=part>Second Part</h1>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
</body>
EOT
		# Test ToC
	ok($output, <<EOT);
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#part-A>First Part</a>
</ul>
<ul>
   <li><a href=#h-1>Chapter</a>
   <ul>
      <li><a href=#h-1.1>Paragraph</a>
   </ul>
</ul>
<ul>
   <li><a href=#part-B>Second Part</a>
</ul>
<ul>
   <li><a href=#h-2>Chapter</a>
   <ul>
      <li><a href=#h-2.1>Paragraph</a>
   </ul>
</ul>
<!-- End of generated Table of Contents -->

   <a name=part-A><h1 class=part>A &nbsp;First Part</h1></a>
   <a name=h-1><h1>Chapter</h1></a>
   <a name=h-1.1><h2>Paragraph</h2></a>
   <a name=part-B><h1 class=part>B &nbsp;Second Part</h1></a>
   <a name=h-2><h1>Chapter</h1></a>
   <a name=h-2.1><h2>Paragraph</h2></a>
</body>
EOT
}  # TestMultipleGroupsPart()


#--- TestMultipleTocs() -------------------------------------------------------
# function: Test multiple ToCs

sub TestMultipleTocs() {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<body>
   <h1>Header One</h1>
   <img src=test1.gif alt="First picture">
   <h2>Paragraph One</h2>
   <img src=test2.gif alt="Second picture">
</body>
EOT

		# Create objects
	my $toc1        = HTML::Toc->new();
	my $toc2        = HTML::Toc->new();
	my $tocInsertor = HTML::TocInsertor->new();
	my $output;
	
		# Set ToC options
	$toc2->setOptions({
		'tokenToToc'             => [{
			'groupId'    => 'image',
			'tokenBegin' => '<img alt=@>'
		}],
	});
		# Generate ToC
	$tocInsertor->insertIntoFile(
		[$toc1, $toc2], $filename, {'output' => \$output}
	);
		# Test ToC
	ok($output, <<EOT);
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#h-1>Header One</a>
   <ul>
      <li><a href=#h-1.1>Paragraph One</a>
   </ul>
</ul>
<!-- End of generated Table of Contents -->

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#image-1>First picture</a>
   <li><a href=#image-2>Second picture</a>
</ul>
<!-- End of generated Table of Contents -->

   <a name=h-1><h1>Header One</h1></a>
   <a name=image-1><img src=test1.gif alt="First picture"></a>
   <a name=h-1.1><h2>Paragraph One</h2></a>
   <a name=image-2><img src=test2.gif alt="Second picture"></a>
</body>
EOT
}  # TestMultipleTocs()


#--- TestSpecifyNumberedList() ------------------------------------------------
# function: Test specifying numbered list.

sub TestSpecifyNumberedList {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter</h1>
	<h2>Paragraph</h2>
</body>
EOT

		# Create objects
	my $toc        = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();
	
		# Set ToC options
	$toc->setOptions({
		'templateLevelBegin' => '"<ol>\n"',
		'templateLevelEnd'   => '"</ol>\n"',
	});
		# Generate ToC
	$tocGenerator->generateFromFile($toc, $filename);
		# Test ToC
	ok($toc->format(), <<EOT);

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ol>
   <li><a href=#h-1>Chapter</a>
   <ol>
      <li><a href=#h-1.1>Paragraph</a>
   </ol>
</ol>
<!-- End of generated Table of Contents -->
EOT
}  # TestSpecifyNumberedList()


#--- TestUpdateFile() ---------------------------------------------------------
# function: Test 'HTML::TocUpdator->updateFile()'

sub TestUpdateFile {
		# Assemble test file
	open(FILE, ">$filename") || die "Can't create $filename: $!";
	print FILE <<'EOT'; close(FILE);
<html>
<body><!-- #BeginToc -->
foo
<!-- #EndToc -->
   <!-- #BeginTocAnchorNameBegin -->bar<!-- #EndTocAnchorNameBegin --><h1>
   Chapter
   </h1><!-- #BeginTocAnchorNameEnd -->foo<!-- #EndTocAnchorNameEnd -->
</body>
</html>
EOT

		# Create objects
	my $toc        = HTML::Toc->new();
	my $tocUpdator = HTML::TocUpdator->new();
	my $output;
	
		# Generate ToC
	$tocUpdator->updateFile($toc, $filename, {'output' => \$output});
		# Test ToC
	ok($output, <<EOT);
<html>
<body><!-- #BeginToc -->
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href=#h-1> Chapter </a>
</ul>
<!-- End of generated Table of Contents -->
<!-- #EndToc -->
   <!-- #BeginTocAnchorNameBegin --><a name=h-1><!-- #EndTocAnchorNameBegin --><h1>
   Chapter
   </h1><!-- #BeginTocAnchorNameEnd --></a><!-- #EndTocAnchorNameEnd -->
</body>
</html>
EOT
}  # TestUpdateFile()


#--- TestUsingCSS() -----------------------------------------------------------
# function: Test multiple ToCs

sub TestUsingCSS() {

		# Create objects
	my $toc          = new HTML::Toc;
	my $tocInsertor  = new HTML::TocInsertor;
	my $output;

   $toc->setOptions({
		'templateLevelBegin'   => '"<ol class=toc_$groupId$level>\n"',
		'templateLevelEnd'     => '"</ol>\n"',
		'doNumberToken'        => 1,
      'tokenToToc' => [{
				'groupId'        => 'appendix',
            'tokenBegin'     => '<h1>',
				'numberingStyle' => 'upper-alpha'
			}, {
				'groupId'        => 'appendix',
            'tokenBegin' 	  => '<h2>',
            'level'      	  => 2,
         }]
   });
	$tocInsertor->insert($toc, <<EOT);
<html>
<head>
   <style type="text/css">
      ol.toc_appendix1 { list-style-type: upper-alpha }
   </style>
</head>
<body>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
</body>
</html>
EOT
		# Insert ToC
	$tocInsertor->insert($toc, <<EOT, {'output' => \$output});
<html>
<head>
   <style type="text/css">
      ol.toc_appendix1 { list-style-type: upper-alpha }
   </style>
</head>
<body>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
</body>
</html>
EOT
		# Test ToC
	ok($output, <<EOT);
<html>
<head>
   <style type="text/css">
      ol.toc_appendix1 { list-style-type: upper-alpha }
   </style>
</head>
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ol class=toc_appendix1>
   <li><a href=#appendix-A>Appendix</a>
   <ol class=toc_appendix2>
      <li><a href=#appendix-A.1>Appendix Paragraph</a>
   </ol>
   <li><a href=#appendix-B>Appendix</a>
   <ol class=toc_appendix2>
      <li><a href=#appendix-B.1>Appendix Paragraph</a>
   </ol>
</ol>
<!-- End of generated Table of Contents -->

   <a name=appendix-A><h1>A &nbsp;Appendix</h1></a>
   <a name=appendix-A.1><h2>A.1 &nbsp;Appendix Paragraph</h2></a>
   <a name=appendix-B><h1>B &nbsp;Appendix</h1></a>
   <a name=appendix-B.1><h2>B.1 &nbsp;Appendix Paragraph</h2></a>
</body>
</html>
EOT
}  # TestUsingCSS()


	# Test 'extendFromFile()'
TestExtendFromFile();
	# Test 'generateFromFile()'
TestGenerateFromFile();
	# Test 'generateFromFiles()'
TestGenerateFromFiles();
	# Test 'doUseGroupsGlobal = 0'
TestGlobalGroups0();
	# Test 'doUseGroupsGlobal = 1'
TestGlobalGroups1();
	# Test 'tocInsertor->insertIntoFile'
TestInsertIntoFile();
	# Test 'tocUpdator->insertIntoFile'
TestInsertIntoFileUsingTocUpdator();
	# Test additional 'appendix' group
TestMultipleGroupsAppendix();
	# Test additional 'part' group
TestMultipleGroupsPart();
	# Test multiple ToCs
TestMultipleTocs();
	# Test specifying numbered list
TestSpecifyNumberedList();
	# Test 'updateFile()'
TestUpdateFile();
	# Test using CSS
TestUsingCSS();
