//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/Mask/DetectorMask.cpp
//! @brief     Implements class DetectorMask.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Device/Mask/DetectorMask.h"
#include "Base/Axis/Scale.h"
#include "Device/Data/Datafield.h"
#include "Device/Mask/IShape2D.h"

MaskPattern::MaskPattern(IShape2D* shape_, bool doMask_)
    : shape(shape_)
    , doMask(doMask_)
{
}
MaskPattern::~MaskPattern()
{
    delete shape;
}
MaskPattern* MaskPattern::clone() const
{
    return new MaskPattern(shape->clone(), doMask);
}

DetectorMask::DetectorMask(const Scale& xAxis, const Scale& yAxis)
    : m_xAxis(xAxis.clone())
    , m_yAxis(yAxis.clone())
    , m_masked(xAxis.size() * yAxis.size(), false)
{
}

DetectorMask::~DetectorMask() = default;

DetectorMask::DetectorMask(const DetectorMask& other)
    : m_xAxis(other.m_xAxis->clone())
    , m_yAxis(other.m_yAxis->clone())
    , m_stack(other.m_stack)
    , m_masked(other.m_masked)
    , m_number_of_masked_channels(other.m_number_of_masked_channels)
{
}

void DetectorMask::addMask(const IShape2D& shape, bool mask_value)
{
    m_stack.emplace_back(new MaskPattern(shape.clone(), mask_value));
    process_masks();
}

bool DetectorMask::isMasked(size_t i_flat) const
{
    return m_number_of_masked_channels == 0 ? false : m_masked[i_flat];
}

bool DetectorMask::hasMasks() const
{
    return !m_stack.empty();
}

size_t DetectorMask::numberOfMasks() const
{
    return m_stack.size();
}

const MaskPattern* DetectorMask::patternAt(size_t iMask) const
{
    return m_stack.at(iMask);
}

void DetectorMask::process_masks()
{
    m_number_of_masked_channels = 0;
    m_masked.clear();
    m_masked.resize(m_xAxis->size() * m_yAxis->size(), false);

    if (m_stack.empty())
        return;

    m_number_of_masked_channels = 0;
    for (size_t i_flat = 0; i_flat < m_masked.size(); ++i_flat) {
        Bin1D binx = m_xAxis->bin((i_flat / m_yAxis->size()) % m_xAxis->size());
        Bin1D biny = m_yAxis->bin(i_flat % m_yAxis->size());
        // setting mask to the data starting from last shape added
        bool is_masked(false);
        for (int k = m_stack.size() - 1; k >= 0; --k) {
            const MaskPattern* const pat = m_stack[k];
            if (pat->shape->contains(binx, biny)) {
                if (pat->doMask)
                    is_masked = true;
                m_masked[i_flat] = pat->doMask;
                break; // i_flat is covered by the shape, stop looking further
            }
        }
        if (is_masked)
            ++m_number_of_masked_channels;
    }
}
