//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Import/ImportDataView.cpp
//! @brief     Implements class ImportDataView
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Import/ImportDataView.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/View/Import/RealDataSelectorWidget.h"
#include "GUI/View/Tool/mainwindow_constants.h"
#include <QAction>
#include <QSplitter>
#include <QVBoxLayout>

ImportDataView::ImportDataView(QWidget* parent, ProjectDocument* document)
    : QWidget(parent)
    , m_selectorWidget(new RealDataSelectorWidget(this, document))
    , m_stackedWidget(new ItemStackPresenter<RealDataPresenter>(true /* reuse widget */))
    , m_document(document)
{
    auto* mainLayout = new QVBoxLayout;
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->setSpacing(0);
    mainLayout->setContentsMargins(0, 0, 0, 0);

    m_stackedWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_selectorWidget->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);
    m_stackedWidget->setSizeHint(QSize(1024, 1024));

    auto* splitter = new QSplitter;
    splitter->addWidget(m_selectorWidget);
    splitter->addWidget(m_stackedWidget);
    splitter->setCollapsible(0, false);
    splitter->setCollapsible(1, false);

    splitter->setSizes(QList<int>() << GUI::Constants::ITEM_SELECTOR_WIDGET_WIDTH
                                    << GUI::Constants::ITEM_SELECTOR_WIDGET_WIDTH * 7);

    mainLayout->addWidget(splitter);

    setLayout(mainLayout);

    updateFunctionalityNarrowing();

    connect(m_selectorWidget, &RealDataSelectorWidget::selectionChanged, this,
            &ImportDataView::onSelectionChanged);

    connect(m_document, &ProjectDocument::modifiedStateChanged, this,
            &ImportDataView::updateFunctionalityNarrowing);

    if (m_selectorWidget->currentItem())
        onSelectionChanged(m_selectorWidget->currentItem());
}

void ImportDataView::onSelectionChanged(RealItem* realItem)
{
    m_stackedWidget->setItem(realItem);
}

void ImportDataView::updateFunctionalityNarrowing()
{
    m_selectorWidget->m_import1dDataAction->setVisible(
        m_document->functionalities().testFlag(ProjectDocument::Specular));

    m_selectorWidget->m_import2dDataAction->setVisible(
        m_document->functionalities().testFlag(ProjectDocument::Gisas)
        || m_document->functionalities().testFlag(ProjectDocument::Offspec)
        || m_document->functionalities().testFlag(ProjectDocument::Depthprobe));
}
