//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Plot2D/IntensityDataWidget.cpp
//! @brief     Implements class IntensityDataWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Plot2D/IntensityDataWidget.h"
#include "GUI/Model/Data/IntensityDataItem.h"
#include "GUI/Support/Util/ActionFactory.h"
#include "GUI/View/Common/IntensityDataPropertyWidget.h"
#include "GUI/View/Plot2D/IntensityDataCanvas.h"
#include "GUI/View/Plot2D/IntensityDataFFTPresenter.h"
#include <QBoxLayout>
#include <QMenu>

IntensityDataWidget::IntensityDataWidget(QWidget* parent)
    : DataAccessWidget(parent)
    , m_intensityCanvas(new IntensityDataCanvas(this))
    , m_propertyWidget(new IntensityDataPropertyWidget(this))
    , m_fftPresenter(new IntensityDataFFTPresenter(this))
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    m_propertyWidget->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Minimum);

    auto* hlayout = new QHBoxLayout(this);
    hlayout->setContentsMargins(0, 0, 0, 0);
    hlayout->setSpacing(0);
    hlayout->addWidget(m_intensityCanvas);
    hlayout->addWidget(m_propertyWidget);

    connect(m_intensityCanvas, &IntensityDataCanvas::customContextMenuRequested, this,
            &IntensityDataWidget::onContextMenuRequest);

    connect(m_fftPresenter, &IntensityDataFFTPresenter::fftActionRequest, this,
            &IntensityDataWidget::onFFTAction);

    m_togglePropertiesAction =
        ActionFactory::createTogglePropertiesPanelAction(this, m_propertyWidget);

    m_propertyWidget->setVisible(false);
}

void IntensityDataWidget::setJobOrRealItem(QObject* job_or_real_item)
{
    JobItem* oldJob = jobItem();
    DataAccessWidget::setJobOrRealItem(job_or_real_item);
    m_intensityCanvas->setJobOrRealItem(job_or_real_item);
    m_propertyWidget->setJobOrRealItem(job_or_real_item);

    if (oldJob != jobItem())
        m_fftPresenter->reset();
    onFFTAction();
}

QList<QAction*> IntensityDataWidget::actionList()
{
    return m_intensityCanvas->actionList() + m_fftPresenter->actionList()
           + QList<QAction*>{m_togglePropertiesAction};
}

void IntensityDataWidget::onContextMenuRequest(const QPoint& point)
{
    QMenu menu;
    for (auto* action : actionList())
        menu.addAction(action);
    menu.exec(point);
}

void IntensityDataWidget::onFFTAction()
{
    if (!currentIntensityDataItem() || !currentIntensityDataItem()->c_field())
        return;

    if (m_fftPresenter->inFFTMode()) {
        auto* fftItem = m_fftPresenter->fftItem(currentIntensityDataItem());
        m_intensityCanvas->setFourierItem(fftItem);
        m_propertyWidget->setFourierItem(fftItem);
    } else {
        // returning ColorMap to non-fft presentation
        m_intensityCanvas->setFourierItem(nullptr);
        m_propertyWidget->setFourierItem(nullptr);
    }
}
