//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/SelectionContainerForm.h
//! @brief     Defines class SelectionContainerForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SELECTIONCONTAINERFORM_H
#define BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SELECTIONCONTAINERFORM_H

#include "GUI/Support/Util/CustomEventFilters.h"
#include "GUI/View/SampleDesigner/LayerEditorUtil.h"
#include "GUI/View/SampleDesigner/SampleEditorController.h"
#include "GUI/View/Tool/LayoutUtil.h"
#include <QComboBox>
#include <QGridLayout>

//! Abstract widget base class to contain a selection, defined by a SelectionDescriptor.
//!
//! This abstract base class contains only the selection combo box and the infrastructure to
//! call the SampleEditorController to change the current selection. A derived class has to
//! implement the showing of the contents (method createContent).
class ISelectionContainerForm : public QWidget {
public:
    virtual void createContent() = 0;

protected:
    ISelectionContainerForm(QWidget* parent, SampleEditorController* ec)
        : QWidget(parent)
        , m_ec(ec)
    {
    }

    template <typename T>
    void initUI(SelectionProperty<T>& d)
    {
        m_gridLayout = new QGridLayout(this);
        m_gridLayout->setContentsMargins(0, 0, 0, 0);
        m_gridLayout->setSpacing(6);

        m_combo = new QComboBox;
        WheelEventEater::install(m_combo);
        m_combo->addItems(d.options());
        m_combo->setCurrentIndex(d.currentIndex());
        m_combo->setMaxVisibleItems(m_combo->count());

        QObject::connect(m_combo, QOverload<int>::of(&QComboBox::currentIndexChanged),
                         [this, &d](int current) {
                             clear();
                             m_ec->setCurrentIndex(this, current, d);
                         });

        m_gridLayout->addWidget(m_combo, 1, 0);
        createContent();
    }

private:
    //! Remove all properties from the layout
    void clear()
    {
        auto* layoutItemOfComboBox = m_gridLayout->itemAtPosition(1, 0);
        m_gridLayout->takeAt(m_gridLayout->indexOf(layoutItemOfComboBox));
        GUI::Util::Layout::clearLayout(m_gridLayout, true);
        m_gridLayout->addWidget(layoutItemOfComboBox->widget(), 1, 0);
    }

protected:
    QGridLayout* m_gridLayout;
    QComboBox* m_combo;
    SampleEditorController* m_ec;
};

//! A widget to contain a selection, defined by a SelectionProperty.
//!
//! This SelectionContainerForm is limited to contain the selection combo box and a list of double
//! values represented by DoubleProperties. The list of DoubleProperties is queried by calling
//! LayerEditorUtil::doublePropertiesOfItem(). To have the correct DoubleProperties on this form,
//! you may have to overload this method according to your class and your needs. The overload will
//! expect the template type you defined for SelectionProperty.
//!
//! Example:
//! SelectionProperty<RotationItemCatalog>  =>
//! LayerEditorUtil::doublePropertiesOfItem(RotationItem*)
//!
//! The connection from selection combo -> SelectionProperty is made via
//! SampleEditorController::setCurrentIndex(), where a command can be used for undo purposes.
//!
//! For each DoubleProperty, a unit-aware DoubleSpinBox is created. The connection from each
//! spinbox to the DoubleProperty is made via SampleEditorController::setDouble(), where a command
//! can be used for undo purposes.
//!
//! If a more complex selection shall be realized (e.g. with
//! sub-selections or different value types), this class is not sufficient.
class SelectionContainerForm : public ISelectionContainerForm {
public:
    template <typename T>
    SelectionContainerForm(QWidget* parent, SelectionProperty<T>& d, SampleEditorController* ec)
        : ISelectionContainerForm(parent, ec)
    {
        currentValues = [&d] { return LayerEditorUtil::doublePropertiesOfItem(d.currentItem()); };
        initUI(d);
    }

    virtual void createContent()
    {
        if (currentValues != nullptr)
            LayerEditorUtil::addMultiPropertyToGrid(m_gridLayout, 1, currentValues(), m_ec, true);
    }

private:
    std::function<DoubleProperties()> currentValues = nullptr;
};

#endif // BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SELECTIONCONTAINERFORM_H
