#!/usr/bin/env python3
# vim:fileencoding=utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

__license__ = 'GPL v3'
__copyright__ = '2014, Kovid Goyal <kovid at kovidgoyal.net>'

# A database of locale data for all the languages known to glibc
# To regenerate run:
# sudo locale-gen -A && python3 lc_time.py

import locale, os, pprint, sys

def generate_data():
    def nl(code):
        return locale.nl_langinfo(code)
    ans = []
    for x, limit in (('day', 8), ('mon', 13)):
        for attr in ('ab' + x, x):
            ans.append((attr, tuple(map(nl, (getattr(locale, '%s_%d' % (attr.upper(), i)) for i in range(1, limit)))))),
    for x in ('d_t_fmt', 'd_fmt', 't_fmt', 't_fmt_ampm', 'radixchar', 'thousep', 'yesexpr', 'noexpr'):
        ans.append((x, nl(getattr(locale, x.upper()))))
    return ans

def main():
    if sys.version_info[0] < 3:
        raise RuntimeError('Must be run using python 3.x')
    locale.setlocale(locale.LC_ALL, '')
    dest = os.path.abspath(__file__)
    os.chdir('/usr/share/i18n/locales')
    data = []
    for f in sorted(os.listdir('.')):
        try:
            locale.setlocale(locale.LC_ALL, (f, 'utf-8'))
        except locale.Error:
            continue
        data.append((f, generate_data()))

    with open(dest, 'r+b') as f:
        raw = f.read()
        marker = b'# The data {{' + b'{'
        pos = raw.find(marker)
        data = pprint.pformat(data, width=160)
        if not isinstance(data, bytes):
            data = data.encode('utf-8')
        f.seek(pos)
        f.truncate()
        f.write(marker + b'\ndata = ' + data + b'\n' + b'# }}' + b'}')


if __name__ == '__main__':
    main()

# The data {{{
data = [('aa_DJ',
  [('abday', ('aca', 'etl', 'tal', 'arb', 'kam', 'gum', 'sab')),
   ('day', ('Acaada', 'Etleeni', 'Talaata', 'Arbaqa', 'Kamiisi', 'Gumqata', 'Sabti')),
   ('abmon', ('qun', 'nah', 'cig', 'agd', 'cax', 'qas', 'qad', 'leq', 'way', 'dit', 'xim', 'kax')),
   ('mon',
    ('Qunxa Garablu',
     'Kudo',
     'Ciggilta Kudo',
     'Agda Baxisso',
     'Caxah Alsa',
     'Qasa Dirri',
     'Qado Dirri',
     'Liiqen',
     'Waysu',
     'Diteli',
     'Ximoli',
     'Kaxxa Garablu')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[oOyY].*'),
   ('noexpr', '^[mnMN].*')]),
 ('aa_ER',
  [('abday', ('Aca', 'Etl', 'Tal', 'Arb', 'Kam', 'Gum', 'Sab')),
   ('day', ('Acaada', 'Etleeni', 'Talaata', 'Arbaqa', 'Kamiisi', 'Gumqata', 'Sabti')),
   ('abmon', ('Qun', 'Nah', 'Cig', 'Agd', 'Cax', 'Qas', 'Qad', 'Leq', 'Way', 'Dit', 'Xim', 'Kax')),
   ('mon',
    ('Qunxa Garablu',
     'Naharsi Kudo',
     'Ciggilta Kudo',
     'Agda Baxisso',
     'Caxah Alsa',
     'Qasa Dirri',
     'Qado Dirri',
     'Leqeeni',
     'Waysu',
     'Diteli',
     'Ximoli',
     'Kaxxa Garablu')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[mnMN].*')]),
 ('aa_ET',
  [('abday', ('Aca', 'Etl', 'Tal', 'Arb', 'Kam', 'Gum', 'Sab')),
   ('day', ('Acaada', 'Etleeni', 'Talaata', 'Arbaqa', 'Kamiisi', 'Gumqata', 'Sabti')),
   ('abmon', ('Qun', 'Kud', 'Cig', 'Agd', 'Cax', 'Qas', 'Qad', 'Leq', 'Way', 'Dit', 'Xim', 'Kax')),
   ('mon',
    ('Qunxa Garablu',
     'Kudo',
     'Ciggilta Kudo',
     'Agda Baxisso',
     'Caxah Alsa',
     'Qasa Dirri',
     'Qado Dirri',
     'Liiqen',
     'Waysu',
     'Diteli',
     'Ximoli',
     'Kaxxa Garablu')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[mnMN].*')]),
 ('af_ZA',
  [('abday', ('So', 'Ma', 'Di', 'Wo', 'Do', 'Vr', 'Sa')),
   ('day', ('Sondag', 'Maandag', 'Dinsdag', 'Woensdag', 'Donderdag', 'Vrydag', 'Saterdag')),
   ('abmon', ('Jan', 'Feb', 'Mrt', 'Apr', 'Mei', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Des')),
   ('mon', ('Januarie', 'Februarie', 'Maart', 'April', 'Mei', 'Junie', 'Julie', 'Augustus', 'September', 'Oktober', 'November', 'Desember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[jJyY]'),
   ('noexpr', '^[nN]')]),
 ('ak_GH',
  [('abday', ('Kwe', 'Dwo', 'Ben', 'Wuk', 'Yaw', 'Fia', 'Mem')),
   ('day', ('Kwesida', 'Dwowda', 'Benada', 'Wukuda', 'Yawda', 'Fida', 'Memeneda')),
   ('abmon', ('S-Ɔ', 'K-Ɔ', 'E-Ɔ', 'E-O', 'E-K', 'O-A', 'A-K', 'D-Ɔ', 'F-Ɛ', 'Ɔ-A', 'Ɔ-O', 'M-Ɔ')),
   ('mon',
    ('Sanda-Ɔpɛpɔn',
     'Kwakwar-Ɔgyefuo',
     'Ebɔw-Ɔbenem',
     'Ebɔbira-Oforisuo',
     'Esusow Aketseaba-Kɔtɔnimba',
     'Obirade-Ayɛwohomumu',
     'Ayɛwoho-Kitawonsa',
     'Difuu-Ɔsandaa',
     'Fankwa-Ɛbɔ',
     'Ɔbɛsɛ-Ahinime',
     'Ɔberɛfɛw-Obubuo',
     'Mumu-Ɔpɛnimba')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%Y/%m/%d'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[dDnN].*')]),
 ('am_ET',
  [('abday', ('እሑድ', 'ሰኞ ', 'ማክሰ', 'ረቡዕ', 'ሐሙስ', 'ዓርብ', 'ቅዳሜ')),
   ('day', ('እሑድ', 'ሰኞ', 'ማክሰኞ', 'ረቡዕ', 'ሐሙስ', 'ዓርብ', 'ቅዳሜ')),
   ('abmon', ('ጃንዩ', 'ፌብሩ', 'ማርች', 'ኤፕረ', 'ሜይ ', 'ጁን ', 'ጁላይ', 'ኦገስ', 'ሴፕቴ', 'ኦክተ', 'ኖቬም', 'ዲሴም')),
   ('mon', ('ጃንዩወሪ', 'ፌብሩወሪ', 'ማርች', 'ኤፕረል', 'ሜይ', 'ጁን', 'ጁላይ', 'ኦገስት', 'ሴፕቴምበር', 'ኦክተውበር', 'ኖቬምበር', 'ዲሴምበር')),
   ('d_t_fmt', '%A፣ %B %e ቀን %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('an_ES',
  [('abday', ('dom', 'lun', 'mar', 'mie', 'chu', 'bie', 'sab')),
   ('day', ('domingo', 'luns', 'martes', 'miecols', 'chuebes', 'biernes', 'sabado')),
   ('abmon', ('chi', 'fre', 'mar', 'abr', 'may', 'chn', 'chl', 'ago', 'set', 'oct', 'nob', 'abi')),
   ('mon', ('chinero', 'frebero', 'marzo', 'abril', 'mayo', 'chunio', 'chulio', 'agosto', 'setiembre', 'octubre', 'nobiembre', 'abiento')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('anp_IN',
  [('abday', ('रवि ', 'सोम ', 'मंगल ', 'बुध ', 'बृहस्पति ', 'शुक्र ', 'शनि ')),
   ('day', ('रविवार ', 'सोमवार ', 'मंगलवार ', 'बुधवार ', 'बृहस्पतिवार ', 'शुक्रवार ', 'शनिवार ')),
   ('abmon', ('जनवरी', 'फरवरी', 'मार्च', 'अप्रैल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितंबर', 'अक्टूबर', 'नवंबर', 'दिसंबर')),
   ('mon', ('जनवरी', 'फरवरी', 'मार्च', 'अप्रैल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितंबर', 'अक्टूबर', 'नवंबर', 'दिसंबर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[हवyY]'),
   ('noexpr', '^[नइnN]')]),
 ('ar_AE',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت ')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_BH',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_DZ',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_EG',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_IN',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%A %d %B %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %B %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_IQ',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_JO',
  [('abday', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('كانون الثاني', 'شباط', 'آذار', 'نيسان', 'نوار', 'حزيران', 'تموز', 'آب', 'أيلول', 'تشرين الأول', 'تشرين الثاني', 'كانون الأول')),
   ('mon', ('كانون الثاني', 'شباط', 'آذار', 'نيسان', 'نوار', 'حزيران', 'تموز', 'آب', 'أيلول', 'تشرين الأول', 'تشرين الثاني', 'كانون الأول')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_KW',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_LB',
  [('abday', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('كانون الثاني', 'شباط', 'آذار', 'نيسان', 'نوار', 'حزيران', 'تموز', 'آب', 'أيلول', 'تشرين الأول', 'تشرين الثاني', 'كانون الأول')),
   ('mon', ('كانون الثاني', 'شباط', 'آذار', 'نيسان', 'نوار', 'حزيران', 'تموز', 'آب', 'أيلول', 'تشرين الأول', 'تشرين الثاني', 'كانون الأول')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_LY',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_MA',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_OM',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_QA',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_SA',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('الأحد', 'الإثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعـة', 'السبت')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('كانون الثاني', 'شباط', 'آذار', 'نيسـان', 'أيار', 'حزيران', 'تـمـوز', 'آب', 'أيلول', 'تشرين الأول', 'تشرين الثاني', 'كانون الأول')),
   ('d_t_fmt', '%A %e %B %Y  %k:%M:%S'),
   ('d_fmt', '%A %e %B %Y'),
   ('t_fmt', '%k:%M:%S'),
   ('t_fmt_ampm', '%k:%M:%S'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_SD',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_SS',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_SY',
  [('abday', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('كانون الثاني', 'شباط', 'آذار', 'نيسان', 'نوار', 'حزيران', 'تموز', 'آب', 'أيلول', 'تشرين الأول', 'تشرين الثاني', 'كانون الأول')),
   ('mon', ('كانون الثاني', 'شباط', 'آذار', 'نيسان', 'نواران', 'حزير', 'تموز', 'آب', 'أيلول', 'تشرين الأول', 'تشرين الثاني', 'كانون الأول')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_TN',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('ar_YE',
  [('abday', ('ح', 'ن', 'ث', 'ر', 'خ', 'ج', 'س')),
   ('day', ('الأحد', 'الاثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت')),
   ('abmon', ('ينا', 'فبر', 'مار', 'أبر', 'ماي', 'يون', 'يول', 'أغس', 'سبت', 'أكت', 'نوف', 'ديس')),
   ('mon', ('يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر')),
   ('d_t_fmt', '%d %b, %Y %Z %I:%M:%S %p'),
   ('d_fmt', '%d %b, %Y'),
   ('t_fmt', '%Z %I:%M:%S '),
   ('t_fmt_ampm', '%Z %I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('as_IN',
  [('abday', ('দেও', 'সোম', 'মঙ্গল', 'বুধ', 'বৃহষ্পতি', 'শুক্ৰ', 'শনি')),
   ('day', ('দেওবাৰ', 'সোমবাৰ', 'মঙ্গলবাৰ', 'বুধবাৰ', 'বৃহষ্পতিবাৰ', 'শুক্ৰবাৰ', 'শনিবাৰ')),
   ('abmon', ('জানুৱাৰী', 'ফেব্ৰুৱাৰী', 'মাৰ্চ', 'এপ্ৰিল', 'মে', 'জুন', 'জুলাই', 'আগ', 'চেপ্তেম্বৰ', 'অক্টোবৰ', 'নভেম্বৰ', 'ডিচেম্বৰ')),
   ('mon', ('জানুৱাৰী', 'ফেব্ৰুৱাৰী', 'মাৰ্চ', 'এপ্ৰিল', 'মে', 'জুন', 'জুলাই', 'আগষ্ট', 'চেপ্তেম্বৰ', 'অক্টোবৰ', 'নভেম্বৰ', 'ডিচেম্বৰ')),
   ('d_t_fmt', '%e %B, %Y %I.%M.%S %p %Z'),
   ('d_fmt', '%e-%m-%Y'),
   ('t_fmt', '%I.%M.%S %p'),
   ('t_fmt_ampm', '%I.%M.%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYহ].*'),
   ('noexpr', '^[nNন].*')]),
 ('ast_ES',
  [('abday', ('dom', 'llu', 'mar', 'mié', 'xue', 'vie', 'sáb')),
   ('day', ('domingu', 'llunes', 'martes', 'miércoles', 'xueves', 'vienres', 'sábadu')),
   ('abmon', ('xin', 'feb', 'mar', 'abr', 'may', 'xun', 'xnt', 'ago', 'set', 'och', 'pay', 'avi')),
   ('mon', ('xineru', 'febreru', 'marzu', 'abril', 'mayu', 'xunu', 'xunetu', 'agostu', 'setiembre', 'ochobre', 'payares', 'avientu')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ayc_PE',
  [('abday', ('tum', 'lun', 'mar', 'mir', 'juy', 'wir', 'saw')),
   ('day', ('tuminku', 'lunisa', 'martisa', 'mirkulisa', 'juywisa', 'wirnisa', 'sawäru')),
   ('abmon', ('ini', 'phi', 'mar', 'awr', 'may', 'jun', 'jul', 'awu', 'sit', 'ukt', 'nuw', 'ris')),
   ('mon', ('inïru', 'phiwriru', 'marsu', 'awrila', 'mayu', 'junyu', 'julyu', 'awustu', 'sitimri', 'uktuwri', 'nuwimri', 'risimri')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[uUsSyY].*'),
   ('noexpr', '^[jJnN].*')]),
 ('az_AZ',
  [('abday', ('baz', 'ber', 'çax', 'çər', 'cax', 'cüm', 'şnb')),
   ('day', ('bazar günü', 'bazar ertəsi', 'çərşənbə axşamı', 'çərşənbə', 'cümə axşamı', 'cümə', 'şənbə')),
   ('abmon', ('Yan', 'Fev', 'Mar', 'Apr', 'May', 'İyn', 'İyl', 'Avq', 'Sen', 'Okt', 'Noy', 'Dek')),
   ('mon', ('yanvar', 'fevral', 'mart', 'aprel', 'may', 'iyun', 'iyul', 'avqust', 'sentyabr', 'oktyabr', 'noyabr', 'dekabr')),
   ('d_t_fmt', '%A, %d %B %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[Bb].*'),
   ('noexpr', '^[YyNn].*')]),
 ('be_BY',
  [('abday', ('Няд', 'Пан', 'Аўт', 'Срд', 'Чцв', 'Пят', 'Суб')),
   ('day', ('Нядзеля', 'Панядзелак', 'Аўторак', 'Серада', 'Чацвер', 'Пятніца', 'Субота')),
   ('abmon', ('Стд', 'Лют', 'Сак', 'Крс', 'Тра', 'Чэр', 'Ліп', 'Жнв', 'Врс', 'Кст', 'Ліс', 'Снж')),
   ('mon', ('Студзень', 'Люты', 'Сакавік', 'Красавік', 'Травень', 'Чэрвень', 'Ліпень', 'Жнівень', 'Верасень', 'Кастрычнік', 'Лістапад', 'Снежань')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[ТтYy].*'),
   ('noexpr', '^[НнNn].*')]),
 ('bem_ZM',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Pa Mulungu', 'Palichimo', 'Palichibuli', 'Palichitatu', 'Palichine', 'Palichisano', 'Pachibelushi')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'July', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('Januari', 'Februari', 'Machi', 'Epreo', 'Mei', 'Juni', 'Julai', 'Ogasti', 'Septemba', 'Oktoba', 'Novemba', 'Disemba')),
   ('d_t_fmt', '%a %d %b %Y %R %Z'),
   ('d_fmt', '%m/%d/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYeE].*'),
   ('noexpr', '^[nNaA].*')]),
 ('ber_DZ',
  [('abday', ('baz', 'bir', 'iki', 'üçü', 'dör', 'beş', 'alt')),
   ('day', ('bazar günü', 'birinci gün', 'ikinci gün', 'üçüncü gün', 'dördüncü gün', 'beşinci gün', 'altıncı gün')),
   ('abmon', ('Yan', 'Fev', 'Mar', 'Apr', 'May', 'İyn', 'İyl', 'Avq', 'Sen', 'Okt', 'Noy', 'Dek')),
   ('mon', ('yanvar', 'fevral', 'mart', 'aprel', 'may', 'iyun', 'iyul', 'avqust', 'sentyabr', 'oktyabr', 'noyabr', 'dekabr')),
   ('d_t_fmt', '%A, %d %B %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[Bb].*'),
   ('noexpr', '^[YyNn].*')]),
 ('ber_MA',
  [('abday', ('baz', 'bir', 'iki', 'üçü', 'dör', 'beş', 'alt')),
   ('day', ('bazar günü', 'birinci gün', 'ikinci gün', 'üçüncü gün', 'dördüncü gün', 'beşinci gün', 'altıncı gün')),
   ('abmon', ('Yan', 'Fev', 'Mar', 'Apr', 'May', 'İyn', 'İyl', 'Avq', 'Sen', 'Okt', 'Noy', 'Dek')),
   ('mon', ('yanvar', 'fevral', 'mart', 'aprel', 'may', 'iyun', 'iyul', 'avqust', 'sentyabr', 'oktyabr', 'noyabr', 'dekabr')),
   ('d_t_fmt', '%A, %d %B %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[Bb].*'),
   ('noexpr', '^[YyNn].*')]),
 ('bg_BG',
  [('abday', ('нд', 'пн', 'вт', 'ср', 'чт', 'пт', 'сб')),
   ('day', ('неделя', 'понеделник', 'вторник', 'сряда', 'четвъртък', 'петък', 'събота')),
   ('abmon', ('яну', 'фев', 'мар', 'апр', 'май', 'юни', 'юли', 'авг', 'сеп', 'окт', 'ное', 'дек')),
   ('mon', ('януари', 'февруари', 'март', 'април', 'май', 'юни', 'юли', 'август', 'септември', 'октомври', 'ноември', 'декември')),
   ('d_t_fmt', '%x (%a) %X %Z'),
   ('d_fmt', '%e.%m.%Y'),
   ('t_fmt', '%k,%M,%S'),
   ('t_fmt_ampm', '%l,%M,%S'),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[+1ДдDdYyOo].*'),
   ('noexpr', '^[-0НнNnKk].*')]),
 ('bho_IN',
  [('abday', ('रवि ', 'सोम ', 'मंगल ', 'बुध ', 'गुरु ', 'शुक्र ', 'शनि ')),
   ('day', ('रविवार ', 'सोमवार ', 'मंगलवार ', 'बुधवार ', 'गुरुवार ', 'शुक्रवार ', 'शनिवार ')),
   ('abmon', ('जनवरी', 'फरवरी', 'मार्च', 'अप्रैल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फरवरी', 'मार्च', 'अप्रैल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('bn_BD',
  [('abday', ('রবি', 'সোম', 'মঙ্গল', 'বুধ', 'বৃহঃ', 'শুক্র', 'শনি')),
   ('day', ('রবিবার', 'সোমবার', 'মঙ্গলবার', 'বুধবার', 'বৃহস্পতিবার', 'শুক্রবার', 'শনিবার')),
   ('abmon', ('জানু', 'ফেব্রু', 'মার্চ', 'এপ্রি', 'মে', 'জুন', 'জুল', 'আগ', 'সেপ্টে', 'অক্টো', 'নভে', 'ডিসে')),
   ('mon', ('জানুয়ারি', 'ফেব্রুয়ারি', 'মার্চ', 'এপ্রিল', 'মে', 'জুন', 'জুলাই', 'আগস্ট', 'সেপ্টেম্বর', 'অক্টোবর', 'নভেম্বর', 'ডিসেম্বর')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[হ্যাঁyY]'),
   ('noexpr', '^[নাnN]')]),
 ('bn_IN',
  [('abday', ('রবি', 'সোম', 'মঙ্গল', 'বুধ', 'বৃহস্পতি', 'শুক্র', 'শনি')),
   ('day', ('রবিবার', 'সোমবার', 'মঙ্গলবার', 'বুধবার', 'বৃহস্পতিবার', 'শুক্রবার', 'শনিবার')),
   ('abmon', ('জানুয়ারি', 'ফেব্রুয়ারি', 'মার্চ', 'এপ্রিল', 'মে', 'জুন', 'জুলাই', 'আগস্ট', 'সেপ্টেম্বর', 'অক্টোবর', 'নভেম্বর', 'ডিসেম্বর')),
   ('mon', ('জানুয়ারি', 'ফেব্রুয়ারি', 'মার্চ', 'এপ্রিল', 'মে', 'জুন', 'জুলাই', 'আগস্ট', 'সেপ্টেম্বর', 'অক্টোবর', 'নভেম্বর', 'ডিসেম্বর')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[হ্যাঁyY]'),
   ('noexpr', '^[নাnN]')]),
 ('bo_CN',
  [('abday', ('ཉི་', 'ཟླ་', 'མིར་', 'ལྷག་', 'པུར་', 'སངས་', 'སྤེན་')),
   ('day', ('གཟའ་ཉི་མ་', 'གཟའ་ཟླ་བ་', 'གཟའ་མིག་དམར་', 'གཟའ་ལྷག་ཕ་', 'གཟའ་པུར་བུ་', 'གཟའ་པ་སངས་', 'གཟའ་སྤེན་ཕ་')),
   ('abmon', ('ཟླ་༡', 'ཟླ་༢', 'ཟླ་༣', 'ཟླ་༤', 'ཟླ་༥', 'ཟླ་༦', 'ཟླ་༧', 'ཟླ་༨', 'ཟླ་༩', 'ཟླ་༡༠', 'ཟླ་༡༡', 'ཟླ་༡༢')),
   ('mon',
    ('ཟླ་བ་དང་པ་',
     'ཟླ་བ་གཉིས་པ་',
     'ཟླ་བ་གསུམ་པ་',
     'ཟླ་བ་བཞི་པ་',
     'ཟླ་བ་ལྔ་ཕ་',
     'ཟླ་བ་དྲུག་པ་',
     'ཟླ་བ་བདུནཔ་',
     'ཟླ་བ་བརྒྱད་པ་',
     'ཟླ་བ་དགུ་པ་',
     'ཟླ་བ་བཅུ་པ་',
     'ཟླ་བ་བཅུ་གཅིག་པ་',
     'ཟླ་བ་བཅུ་གཉིས་པ་')),
   ('d_t_fmt', 'པསྱི་ལོ%yཟལ%mཚེས%dཆུ་ཚོད%Hཀསར་མ%Mཀསར་ཆ%S'),
   ('d_fmt', 'པསྱི་ལོ%yཟལ%mཚེས%d'),
   ('t_fmt', 'ཆུ་ཚོད%Hཀསར་མ%Mཀསར་ཆ%S'),
   ('t_fmt_ampm', 'ཆུ་ཚོད%Iཀསར་མ%Mཀསར་ཆ%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ཨYy].*'),
   ('noexpr', '^[མNn].*')]),
 ('bo_IN',
  [('abday', ('ཉི་', 'ཟླ་', 'མིར་', 'ལྷག་', 'པུར་', 'སངས་', 'སྤེན་')),
   ('day', ('གཟའ་ཉི་མ་', 'གཟའ་ཟླ་བ་', 'གཟའ་མིག་དམར་', 'གཟའ་ལྷག་ཕ་', 'གཟའ་པུར་བུ་', 'གཟའ་པ་སངས་', 'གཟའ་སྤེན་ཕ་')),
   ('abmon', ('ཟླ་༡', 'ཟླ་༢', 'ཟླ་༣', 'ཟླ་༤', 'ཟླ་༥', 'ཟླ་༦', 'ཟླ་༧', 'ཟླ་༨', 'ཟླ་༩', 'ཟླ་༡༠', 'ཟླ་༡༡', 'ཟླ་༡༢')),
   ('mon',
    ('ཟླ་བ་དང་པ་',
     'ཟླ་བ་གཉིས་པ་',
     'ཟླ་བ་གསུམ་པ་',
     'ཟླ་བ་བཞི་པ་',
     'ཟླ་བ་ལྔ་ཕ་',
     'ཟླ་བ་དྲུག་པ་',
     'ཟླ་བ་བདུནཔ་',
     'ཟླ་བ་བརྒྱད་པ་',
     'ཟླ་བ་དགུ་པ་',
     'ཟླ་བ་བཅུ་པ་',
     'ཟླ་བ་བཅུ་གཅིག་པ་',
     'ཟླ་བ་བཅུ་གཉིས་པ་')),
   ('d_t_fmt', 'པསྱི་ལོ%yཟལ%mཚེས%dཆུ་ཚོད%Hཀསར་མ%Mཀསར་ཆ%S'),
   ('d_fmt', 'པསྱི་ལོ%yཟལ%mཚེས%d'),
   ('t_fmt', 'ཆུ་ཚོད%Hཀསར་མ%Mཀསར་ཆ%S'),
   ('t_fmt_ampm', 'ཆུ་ཚོད%Iཀསར་མ%Mཀསར་ཆ%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ཨYy].*'),
   ('noexpr', '^[མNn].*')]),
 ('br_FR',
  [('abday', ('sul', 'lun', 'meu', 'mer', 'yao', 'gwe', 'sad')),
   ('day', ('sul', 'lun', 'meurzh', "merc'her", 'yaou', 'gwener', 'sadorn')),
   ('abmon', ('Gen ', "C'hw", 'Meu ', 'Ebr ', 'Mae ', 'Eve ', 'Gou ', 'Eos ', 'Gwe ', 'Her ', 'Du  ', 'Ker ')),
   ('mon', ('Genver', "C'hwevrer", 'Meurzh', 'Ebrel', 'Mae', 'Mezheven', 'Gouere', 'Eost', 'Gwengolo', 'Here', 'Du', 'Kerzu')),
   ('d_t_fmt', "D'ar %A %d a viz %B %Y"),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%Ie%M:%S %p'),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[oOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('brx_IN',
  [('abday', ('रबि', 'सम', 'मंगल', 'बुद', 'बिसथि', 'सुखुर', 'सुनि')),
   ('day', ('रबिबार', 'सोबार', 'मंगलबार', 'बुदबार', 'बिसथिबार', 'सुखुरबार', 'सुनिबार')),
   ('abmon', ('जानुवारी', 'फेब्रुवारी', 'मार्स', 'एप्रिल', 'मे', 'जुन', 'जुलाइ', 'आगस्थ', 'सेबथेज्ब़र', 'अखथबर', 'नबेज्ब़र', 'दिसेज्ब़र')),
   ('mon', ('जानुवारी', 'फेब्रुवारी', 'मार्स', 'एप्रिल', 'मे', 'जुन', 'जुलाइ', 'आगस्थ', 'सेबथेज्ब़र', 'अखथबर', 'नबेज्ब़र', 'दिसेज्ब़र')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^(नंगौ|[yY])'),
   ('noexpr', '^(नङा|[nN])')]),
 ('bs_BA',
  [('abday', ('Ned', 'Pon', 'Uto', 'Sri', 'Čet', 'Pet', 'Sub')),
   ('day', ('Nedjelja', 'Ponedjeljak', 'Utorak', 'Srijeda', 'Četvrtak', 'Petak', 'Subota')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'Maj', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dec')),
   ('mon', ('Januar', 'Februar', 'Mart', 'April', 'Maj', 'Juni', 'Juli', 'August', 'Septembar', 'Oktobar', 'Novembar', 'Decembar')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[dDyY]*.'),
   ('noexpr', '^[nN]*.')]),
 ('byn_ER',
  [('abday', ('ሰ/ቅ', 'ሰኑ', 'ሰሊጝ', 'ለጓ', 'ኣምድ', 'ኣርብ', 'ሰ/ሽ')),
   ('day', ('ሰንበር ቅዳዅ', 'ሰኑ', 'ሰሊጝ', 'ለጓ ወሪ ለብዋ', 'ኣምድ', 'ኣርብ', 'ሰንበር ሽጓዅ')),
   ('abmon', ('ልደት', 'ካብኽ', 'ክብላ', 'ፋጅኺ', 'ክቢቅ', 'ም/ት', 'ኰር', 'ማርያ', 'ያኸኒ', 'መተሉ', 'ም/ም', 'ተሕሳ')),
   ('mon', ('ልደትሪ', 'ካብኽብቲ', 'ክብላ', 'ፋጅኺሪ', 'ክቢቅሪ', 'ምኪኤል ትጓ̅ኒሪ', 'ኰርኩ', 'ማርያም ትሪ', 'ያኸኒ መሳቅለሪ', 'መተሉ', 'ምኪኤል መሽወሪ', 'ተሕሳስሪ')),
   ('d_t_fmt', '%A፡ %B %e ግርጋ %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('ca_AD',
  [('abday', ('dg', 'dl', 'dt', 'dc', 'dj', 'dv', 'ds')),
   ('day', ('diumenge', 'dilluns', 'dimarts', 'dimecres', 'dijous', 'divendres', 'dissabte')),
   ('abmon', ('gen', 'feb', 'mar', 'abr', 'mai', 'jun', 'jul', 'ago', 'set', 'oct', 'nov', 'des')),
   ('mon', ('gener', 'febrer', 'març', 'abril', 'maig', 'juny', 'juliol', 'agost', 'setembre', 'octubre', 'novembre', 'desembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ca_ES',
  [('abday', ('dg', 'dl', 'dt', 'dc', 'dj', 'dv', 'ds')),
   ('day', ('diumenge', 'dilluns', 'dimarts', 'dimecres', 'dijous', 'divendres', 'dissabte')),
   ('abmon', ('gen', 'feb', 'mar', 'abr', 'mai', 'jun', 'jul', 'ago', 'set', 'oct', 'nov', 'des')),
   ('mon', ('gener', 'febrer', 'març', 'abril', 'maig', 'juny', 'juliol', 'agost', 'setembre', 'octubre', 'novembre', 'desembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ca_FR',
  [('abday', ('dg', 'dl', 'dt', 'dc', 'dj', 'dv', 'ds')),
   ('day', ('diumenge', 'dilluns', 'dimarts', 'dimecres', 'dijous', 'divendres', 'dissabte')),
   ('abmon', ('gen', 'feb', 'mar', 'abr', 'mai', 'jun', 'jul', 'ago', 'set', 'oct', 'nov', 'des')),
   ('mon', ('gener', 'febrer', 'març', 'abril', 'maig', 'juny', 'juliol', 'agost', 'setembre', 'octubre', 'novembre', 'desembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ca_IT',
  [('abday', ('dg', 'dl', 'dt', 'dc', 'dj', 'dv', 'ds')),
   ('day', ('diumenge', 'dilluns', 'dimarts', 'dimecres', 'dijous', 'divendres', 'dissabte')),
   ('abmon', ('gen', 'feb', 'mar', 'abr', 'mai', 'jun', 'jul', 'ago', 'set', 'oct', 'nov', 'des')),
   ('mon', ('gener', 'febrer', 'març', 'abril', 'maig', 'juny', 'juliol', 'agost', 'setembre', 'octubre', 'novembre', 'desembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('cmn_TW',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六')),
   ('abmon', (' 1月', ' 2月', ' 3月', ' 4月', ' 5月', ' 6月', ' 7月', ' 8月', ' 9月', '10月', '11月', '12月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%Y年%m月%d日 (%A) %H點%M分%S秒'),
   ('d_fmt', '%Y年%m月%d日'),
   ('t_fmt', '%H點%M分%S秒'),
   ('t_fmt_ampm', '%p %I點%M分%S秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY是]'),
   ('noexpr', '^[nN不否]')]),
 ('crh_UA',
  [('abday', ('Baz', 'Ber', 'Sal', 'Çar', 'Caq', 'Cum', 'Cer')),
   ('day', ('Bazar', 'Bazarertesi', 'Salı', 'Çarşembe', 'Cumaaqşamı', 'Cuma', 'Cumaertesi')),
   ('abmon', ('Yan', 'Fev', 'Mar', 'Apr', 'May', 'İyn', 'İyl', 'Avg', 'Sen', 'Okt', 'Noy', 'Dek')),
   ('mon', ('Yanvar', 'Fevral', 'Mart', 'Aprel', 'Mayıs', 'İyun', 'İyul', 'Avgust', 'Sentâbr', 'Oktâbr', 'Noyabr', 'Dekabr')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[yYeE]'),
   ('noexpr', '^[nNhH]')]),
 ('cs_CZ',
  [('abday', ('Ne', 'Po', 'Út', 'St', 'Čt', 'Pá', 'So')),
   ('day', ('Neděle', 'Pondělí', 'Úterý', 'Středa', 'Čtvrtek', 'Pátek', 'Sobota')),
   ('abmon', ('led', 'úno', 'bře', 'dub', 'kvě', 'čen', 'čec', 'srp', 'zář', 'říj', 'lis', 'pro')),
   ('mon', ('leden', 'únor', 'březen', 'duben', 'květen', 'červen', 'červenec', 'srpen', 'září', 'říjen', 'listopad', 'prosinec')),
   ('d_t_fmt', '%a\xa0%-d.\xa0%B\xa0%Y,\xa0%H:%M:%S\xa0%Z'),
   ('d_fmt', '%-d.%-m.%Y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S'),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[aAyY].*'),
   ('noexpr', '^[nN].*')]),
 ('csb_PL',
  [('abday', ('nie', 'pòn', 'wtó', 'str', 'czw', 'pią', 'sob')),
   ('day', ('niedzela', 'pòniedzôłk', 'wtórk', 'strzoda', 'czwiôrtk', 'piątk', 'sobòta')),
   ('abmon', ('stë', 'gro', 'stm', 'łżë', 'môj', 'cze', 'lëp', 'zél', 'séw', 'ruj', 'lës', 'gòd')),
   ('mon', ('stëcznik', 'gromicznik', 'strumiannik', 'łżëkwiôt', 'môj', 'czerwińc', 'lëpinc', 'zélnik', 'séwnik', 'rujan', 'lëstopadnik', 'gòdnik')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[JjTtYy].*'),
   ('noexpr', '^[nN].*')]),
 ('cv_RU',
  [('abday', ('vr', 'tn', 'yt', 'jn', 'kş', 'er', 'šm')),
   ('day', ('vyrsarnikun', 'tuntikun', 'ytlarikun', 'junkun', 'kĕşnernikun', 'ernekun', 'šămatkun')),
   ('abmon', ('KĂR', 'NAR', 'PUŠ', 'AKA', 'ŞU', 'ŞĔR', 'UTĂ', 'ŞUR', 'AVĂ', 'JUP', 'CÜK', 'RAŠ')),
   ('mon', ('kărlac', 'narăs', 'puš', 'aka', 'şu', 'şĕrtme', 'ută', 'şurla', 'avăn', 'jupa', 'cük', 'raštav')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('cy_GB',
  [('abday', ('Sul', 'Llu', 'Maw', 'Mer', 'Iau', 'Gwe', 'Sad')),
   ('day', ('Sul', 'Llun', 'Mawrth', 'Mercher', 'Iau', 'Gwener', 'Sadwrn')),
   ('abmon', ('Ion', 'Chw', 'Maw', 'Ebr', 'Mai', 'Meh', 'Gor', 'Aws', 'Med', 'Hyd', 'Tach', 'Rha')),
   ('mon', ('Ionawr', 'Chwefror', 'Mawrth', 'Ebrill', 'Mai', 'Mehefin', 'Gorffennaf', 'Awst', 'Medi', 'Hydref', 'Tachwedd', 'Rhagfyr')),
   ('d_t_fmt', 'Dydd %A %d mis %B %Y %T %Z'),
   ('d_fmt', '%d.%m.%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%l:%M:%S %P %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[iItTyY].*'),
   ('noexpr', '^[nN].*')]),
 ('da_DK',
  [('abday', ('søn', 'man', 'tir', 'ons', 'tor', 'fre', 'lør')),
   ('day', ('søndag', 'mandag', 'tirsdag', 'onsdag', 'torsdag', 'fredag', 'lørdag')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'maj', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januar', 'februar', 'marts', 'april', 'maj', 'juni', 'juli', 'august', 'september', 'oktober', 'november', 'december')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[1JjYy].*'),
   ('noexpr', '^[0Nn].*')]),
 ('de_AT',
  [('abday', ('Son', 'Mon', 'Die', 'Mit', 'Don', 'Fre', 'Sam')),
   ('day', ('Sonntag', 'Montag', 'Dienstag', 'Mittwoch', 'Donnerstag', 'Freitag', 'Samstag')),
   ('abmon', ('Jän', 'Feb', 'Mär', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dez')),
   ('mon', ('Jänner', 'Februar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('de_BE',
  [('abday', ('Son', 'Mon', 'Die', 'Mit', 'Don', 'Fre', 'Sam')),
   ('day', ('Sonntag', 'Montag', 'Dienstag', 'Mittwoch', 'Donnerstag', 'Freitag', 'Samstag')),
   ('abmon', ('Jan', 'Feb', 'Mär', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dez')),
   ('mon', ('Januar', 'Februar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('de_CH',
  [('abday', ('Son', 'Mon', 'Die', 'Mit', 'Don', 'Fre', 'Sam')),
   ('day', ('Sonntag', 'Montag', 'Dienstag', 'Mittwoch', 'Donnerstag', 'Freitag', 'Samstag')),
   ('abmon', ('Jan', 'Feb', 'Mär', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dez')),
   ('mon', ('Januar', 'Februar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', "'"),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('de_DE',
  [('abday', ('So', 'Mo', 'Di', 'Mi', 'Do', 'Fr', 'Sa')),
   ('day', ('Sonntag', 'Montag', 'Dienstag', 'Mittwoch', 'Donnerstag', 'Freitag', 'Samstag')),
   ('abmon', ('Jan', 'Feb', 'Mär', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dez')),
   ('mon', ('Januar', 'Februar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('de_LU',
  [('abday', ('Son', 'Mon', 'Die', 'Mit', 'Don', 'Fre', 'Sam')),
   ('day', ('Sonntag', 'Montag', 'Dienstag', 'Mittwoch', 'Donnerstag', 'Freitag', 'Samstag')),
   ('abmon', ('Jan', 'Feb', 'Mär', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dez')),
   ('mon', ('Januar', 'Februar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('doi_IN',
  [('abday', ('ऐत ', 'सोम ', 'मंगल ', 'बुध ', 'बीर ', 'शुक्कर ', 'श्नीचर ')),
   ('day', ('ऐतबार ', 'सोमबार ', 'मंगलबर ', 'बुधबार ', 'बीरबार ', 'शुक्करबार ', 'श्नीचरबार ')),
   ('abmon', ('जनवरी', 'फरवरी', 'मार्च', 'एप्रैल', 'मेई', 'जून', 'जूलै', 'अगस्त', 'सितंबर', 'अक्तूबर', 'नवंबर', 'दिसंबर')),
   ('mon', ('जनवरी', 'फरवरी', 'मार्च', 'एप्रैल', 'मेई', 'जून', 'जूलै', 'अगस्त', 'सितंबर', 'अक्तूबर', 'नवंबर', 'दिसंबर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^(ऑह|[yY])'),
   ('noexpr', '^(ना|[nN])')]),
 ('dv_MV',
  [('abday', ('އާދީއްތަ', 'ހޯމަ', 'އަންގާރަ', 'ބުދަ', 'ބުރާސްފަތި', 'ހުކުރު', 'ހޮނިހިރު')),
   ('day', ('އާދީއްތަ', 'ހޯމަ', 'އަންގާރަ', 'ބުދަ', 'ބުރާސްފަތި', 'ހުކުރު', 'ހޮނިހިރު')),
   ('abmon', ('ޖެނުއަރީ', 'ފެބްރުއަރީ', 'މާރޗް', 'އެޕްރީލް', 'މެއި', 'ޖޫން', 'ޖުލައި', 'އޮގަސްޓް', 'ސެޕްޓެންބަރ', 'އޮކްޓޫބަރ', 'ނޮވެންބަރ', 'ޑިސެންބަރ')),
   ('mon', ('ޖެނުއަރީ', 'ފެބްރުއަރީ', 'މާރޗް', 'އެޕްރީލް', 'މެއި', 'ޖޫން', 'ޖުލައި', 'އޮގަސްޓް', 'ސެޕްޓެންބަރ', 'އޮކްޓޫބަރ', 'ނޮވެންބަރ', 'ޑިސެންބަރ')),
   ('d_t_fmt', '%Z %H:%M:%S %Y %b %d %a'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%P %I:%M:%S'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('dz_BT',
  [('abday', ('ཟླ་', 'མིར་', 'ལྷག་', 'པུར་', 'སངས་', 'སྤེན་', 'ཉི་')),
   ('day', ('གཟའ་ཟླ་བ་', 'གཟའ་མིག་དམར་', 'གཟའ་ལྷག་ཕ་', 'གཟའ་པུར་བུ་', 'གཟའ་པ་སངས་', 'གཟའ་སྤེན་ཕ་', 'གཟའ་ཉི་མ་')),
   ('abmon', ('ཟླ་༡', 'ཟླ་༢', 'ཟླ་༣', 'ཟླ་༤', 'ཟླ་༥', 'ཟླ་༦', 'ཟླ་༧', 'ཟླ་༨', 'ཟླ་༩', 'ཟླ་༡༠', 'ཟླ་༡༡', 'ཟླ་༡༢')),
   ('mon',
    ('ཟླ་བ་དང་པ་',
     'ཟླ་བ་གཉིས་པ་',
     'ཟླ་བ་གསུམ་པ་',
     'ཟླ་བ་བཞི་པ་',
     'ཟླ་བ་ལྔ་ཕ་',
     'ཟླ་བ་དྲུག་པ་',
     'ཟླ་བ་བདུནཔ་',
     'ཟླ་བ་བརྒྱད་པ་',
     'ཟླ་བ་དགུ་པ་',
     'ཟླ་བ་བཅུ་པ་',
     'ཟླ་བ་བཅུ་གཅིག་པ་',
     'ཟླ་བ་བཅུ་གཉིས་པ་')),
   ('d_t_fmt', 'པསྱི་ལོ%yཟལ%mཚེས%dཆུ་ཚོད%Hཀསར་མ%Mཀསར་ཆ%S'),
   ('d_fmt', 'པསྱི་ལོ%yཟལ%mཚེས%d'),
   ('t_fmt', 'ཆུ་ཚོད%Hཀསར་མ%Mཀསར་ཆ%S'),
   ('t_fmt_ampm', 'ཆུ་ཚོད%Iཀསར་མ%Mཀསར་ཆ%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ཨYy].*'),
   ('noexpr', '^[མNn].*')]),
 ('el_CY',
  [('abday', ('Κυρ', 'Δευ', 'Τρι', 'Τετ', 'Πεμ', 'Παρ', 'Σαβ')),
   ('day', ('Κυριακή', 'Δευτέρα', 'Τρίτη', 'Τετάρτη', 'Πέμπτη', 'Παρασκευή', 'Σάββατο')),
   ('abmon', ('Ιαν', 'Φεβ', 'Μάρ', 'Απρ', 'Μάι', 'Ιούν', 'Ιούλ', 'Αύγ', 'Σεπ', 'Οκτ', 'Νοέ', 'Δεκ')),
   ('mon',
    ('Ιανουάριος', 'Φεβρουάριος', 'Μάρτιος', 'Απρίλιος', 'Μάιος', 'Ιούνιος', 'Ιούλιος', 'Αύγουστος', 'Σεπτέμβριος', 'Οκτώβριος', 'Νοέμβριος', 'Δεκέμβριος')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[νΝyY].*'),
   ('noexpr', '^[οΟnN].*')]),
 ('el_GR',
  [('abday', ('Κυρ', 'Δευ', 'Τρι', 'Τετ', 'Πεμ', 'Παρ', 'Σαβ')),
   ('day', ('Κυριακή', 'Δευτέρα', 'Τρίτη', 'Τετάρτη', 'Πέμπτη', 'Παρασκευή', 'Σάββατο')),
   ('abmon', ('Ιαν', 'Φεβ', 'Μάρ', 'Απρ', 'Μάι', 'Ιούν', 'Ιούλ', 'Αύγ', 'Σεπ', 'Οκτ', 'Νοέ', 'Δεκ')),
   ('mon',
    ('Ιανουάριος', 'Φεβρουάριος', 'Μάρτιος', 'Απρίλιος', 'Μάιος', 'Ιούνιος', 'Ιούλιος', 'Αύγουστος', 'Σεπτέμβριος', 'Οκτώβριος', 'Νοέμβριος', 'Δεκέμβριος')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[νΝyY].*'),
   ('noexpr', '^[οΟnN].*')]),
 ('en_AG',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%l:%M:%S %P %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('en_AU',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('en_BW',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('en_CA',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYoO].*'),
   ('noexpr', '^[nN].*')]),
 ('en_DK',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%Y-%m-%dT%T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[1JjsSyYoO].*'),
   ('noexpr', '^[0nN].*')]),
 ('en_GB',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%l:%M:%S %P %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('en_HK',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%A, %B %d, %Y %p%I:%M:%S %Z'),
   ('d_fmt', '%A, %B %d, %Y'),
   ('t_fmt', '%I:%M:%S %Z'),
   ('t_fmt_ampm', '%p%I:%M:%S %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('en_IE',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('en_IN',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%A %d %B %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %B %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('en_NG',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('en_NZ',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('en_PH',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%A, %d %B, %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A, %d %B, %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('en_SG',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %r'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('en_US',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%m/%d/%Y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('en_ZA',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('en_ZM',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%l:%M:%S %P %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYeE].*'),
   ('noexpr', '^[nNaA].*')]),
 ('en_ZW',
  [('abday', ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat')),
   ('day', ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('es_AR',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_BO',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_CL',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_CO',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_CR',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_CU',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_DO',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_EC',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_ES',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_GT',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_HN',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_MX',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', '\u2009'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_NI',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_PA',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_PE',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_PR',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_PY',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_SV',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_US',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_UY',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('es_VE',
  [('abday', ('dom', 'lun', 'mar', 'mié', 'jue', 'vie', 'sáb')),
   ('day', ('domingo', 'lunes', 'martes', 'miércoles', 'jueves', 'viernes', 'sábado')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('et_EE',
  [('abday', ('P', 'E', 'T', 'K', 'N', 'R', 'L')),
   ('day', ('pühapäev', 'esmaspäev', 'teisipäev', 'kolmapäev', 'neljapäev', 'reede', 'laupäev')),
   ('abmon', ('jaan ', 'veebr', 'märts', 'apr  ', 'mai  ', 'juuni', 'juuli', 'aug  ', 'sept ', 'okt  ', 'nov  ', 'dets ')),
   ('mon', ('jaanuar', 'veebruar', 'märts', 'aprill', 'mai', 'juuni', 'juuli', 'august', 'september', 'oktoober', 'november', 'detsember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[JjYy].*'),
   ('noexpr', '^[EeNn].*')]),
 ('eu_ES',
  [('abday', ('ig.', 'al.', 'ar.', 'az.', 'og.', 'or.', 'lr.')),
   ('day', ('igandea', 'astelehena', 'asteartea', 'asteazkena', 'osteguna', 'ostirala', 'larunbata')),
   ('abmon', ('urt', 'ots', 'mar', 'api', 'mai', 'eka', 'uzt', 'abu', 'ira', 'urr', 'aza', 'abe')),
   ('mon', ('urtarrila', 'otsaila', 'martxoa', 'apirila', 'maiatza', 'ekaina', 'uztaila', 'abuztua', 'iraila', 'urria', 'azaroa', 'abendua')),
   ('d_t_fmt', '%y-%m-%d %T %Z'),
   ('d_fmt', '%a, %Y.eko %bren %da'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[bBsSyY].*'),
   ('noexpr', '^[eEnN].*')]),
 ('fa_IR',
  [('abday', ('یکشنبه', 'دوشنبه', 'سه\u200cشنبه', 'چهارشنبه', 'پنجشنبه', 'جمعه', 'شنبه')),
   ('day', ('یکشنبه', 'دوشنبه', 'سه\u200cشنبه', 'چهارشنبه', 'پنجشنبه', 'جمعه', 'شنبه')),
   ('abmon', ('ژانویه', 'فوریه', 'مارس', 'آوریل', 'مه', 'ژوئن', 'ژوئیه', 'اوت', 'سپتامبر', 'اكتبر', 'نوامبر', 'دسامبر')),
   ('mon', ('ژانویه', 'فوریه', 'مارس', 'آوریل', 'مه', 'ژوئن', 'ژوئیه', 'اوت', 'سپتامبر', 'اكتبر', 'نوامبر', 'دسامبر')),
   ('d_t_fmt', '\u202b%A %Oe %B %Oy، %OH:%OM:%OS\u202c'),
   ('d_fmt', '%Oy/%Om/%Od'),
   ('t_fmt', '%OH:%OM:%OS'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYآبHf].*'),
   ('noexpr', '^[nNخنok].*')]),
 ('ff_SN',
  [('abday', ('dew', 'aaɓ', 'maw', 'nje', 'naa', 'mwd', 'hbi')),
   ('day', ('dewo', 'aaɓnde', 'mawbaare', 'njeslaare', 'naasaande', 'mawnde', 'hoore-biir')),
   ('abmon', ('sii', 'col', 'mbo', 'see', 'duu', 'kor', 'mor', 'juk', 'slt', 'yar', 'jol', 'bow')),
   ('mon', ('siilo', 'colte', 'mbooy', 'seeɗto', 'duujal', 'korse', 'morso', 'juko', 'siilto', 'yarkomaa', 'jolal', 'bowte')),
   ('d_t_fmt', '%a %d %b %Y %R %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%R'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[yYeE].*'),
   ('noexpr', '^[nNaA].*')]),
 ('fi_FI',
  [('abday', ('su', 'ma', 'ti', 'ke', 'to', 'pe', 'la')),
   ('day', ('sunnuntai', 'maanantai', 'tiistai', 'keskiviikko', 'torstai', 'perjantai', 'lauantai')),
   ('abmon', ('tammi', 'helmi', 'maalis', 'huhti', 'touko', 'kesä', 'heinä', 'elo', 'syys', 'loka', 'marras', 'joulu')),
   ('mon', ('tammikuu', 'helmikuu', 'maaliskuu', 'huhtikuu', 'toukokuu', 'kesäkuu', 'heinäkuu', 'elokuu', 'syyskuu', 'lokakuu', 'marraskuu', 'joulukuu')),
   ('d_t_fmt', '%a %e. %Bta %Y %H.%M.%S'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%H.%M.%S'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[KkYy].*'),
   ('noexpr', '^[EeNn].*')]),
 ('fil_PH',
  [('abday', ('Lin', 'Lun', 'Mar', 'Miy', 'Huw', 'Biy', 'Sab')),
   ('day', ('Linggo', 'Lunes', 'Martes', 'Miyerkoles', 'Huwebes', 'Biyernes', 'Sabado')),
   ('abmon', ('Ene', 'Peb', 'Mar', 'Abr', 'May', 'Hun', 'Hul', 'Ago', 'Sep', 'Okt', 'Nob', 'Dis')),
   ('mon', ('Enero', 'Pebrero', 'Marso', 'Abril', 'Mayo', 'Hunyo', 'Hulyo', 'Agosto', 'Septiyembre', 'Oktubre', 'Nobiyembre', 'Disyembre')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%m/%d/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '[oOyY].*'),
   ('noexpr', '[hHnN].*')]),
 ('fo_FO',
  [('abday', ('sun', 'mán', 'týs', 'mik', 'hós', 'frí', 'ley')),
   ('day', ('sunnudagur', 'mánadagur', 'týsdagur', 'mikudagur', 'hósdagur', 'fríggjadagur', 'leygardagur')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'mai', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'des')),
   ('mon', ('januar', 'februar', 'mars', 'apríl', 'mai', 'juni', 'juli', 'august', 'september', 'oktober', 'november', 'desember')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m-%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[JjYy].*'),
   ('noexpr', '^[Nn].*')]),
 ('fr_BE',
  [('abday', ('dim', 'lun', 'mar', 'mer', 'jeu', 'ven', 'sam')),
   ('day', ('dimanche', 'lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi')),
   ('abmon', ('jan', 'fév', 'mar', 'avr', 'mai', 'jun', 'jui', 'aoû', 'sep', 'oct', 'nov', 'déc')),
   ('mon', ('janvier', 'février', 'mars', 'avril', 'mai', 'juin', 'juillet', 'août', 'septembre', 'octobre', 'novembre', 'décembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[oOjJyY1].*'),
   ('noexpr', '^[nN0].*')]),
 ('fr_CA',
  [('abday', ('dim', 'lun', 'mar', 'mer', 'jeu', 'ven', 'sam')),
   ('day', ('dimanche', 'lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi')),
   ('abmon', ('jan', 'fév', 'mar', 'avr', 'mai', 'jun', 'jui', 'aoû', 'sep', 'oct', 'nov', 'déc')),
   ('mon', ('janvier', 'février', 'mars', 'avril', 'mai', 'juin', 'juillet', 'août', 'septembre', 'octobre', 'novembre', 'décembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[oOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('fr_CH',
  [('abday', ('dim', 'lun', 'mar', 'mer', 'jeu', 'ven', 'sam')),
   ('day', ('dimanche', 'lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi')),
   ('abmon', ('jan', 'fév', 'mar', 'avr', 'mai', 'jun', 'jui', 'aoû', 'sep', 'oct', 'nov', 'déc')),
   ('mon', ('janvier', 'février', 'mars', 'avril', 'mai', 'juin', 'juillet', 'août', 'septembre', 'octobre', 'novembre', 'décembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d. %m. %y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', "'"),
   ('yesexpr', '^[OojJsSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('fr_FR',
  [('abday', ('dim.', 'lun.', 'mar.', 'mer.', 'jeu.', 'ven.', 'sam.')),
   ('day', ('dimanche', 'lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi')),
   ('abmon', ('janv.', 'févr.', 'mars', 'avril', 'mai', 'juin', 'juil.', 'août', 'sept.', 'oct.', 'nov.', 'déc.')),
   ('mon', ('janvier', 'février', 'mars', 'avril', 'mai', 'juin', 'juillet', 'août', 'septembre', 'octobre', 'novembre', 'décembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[oOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('fr_LU',
  [('abday', ('dim', 'lun', 'mar', 'mer', 'jeu', 'ven', 'sam')),
   ('day', ('dimanche', 'lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi')),
   ('abmon', ('jan', 'fév', 'mar', 'avr', 'mai', 'jun', 'jui', 'aoû', 'sep', 'oct', 'nov', 'déc')),
   ('mon', ('janvier', 'février', 'mars', 'avril', 'mai', 'juin', 'juillet', 'août', 'septembre', 'octobre', 'novembre', 'décembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[oOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('fur_IT',
  [('abday', ('Dom', 'Lun', 'Mar', 'Mia', 'Joi', 'Vin', 'Sab')),
   ('day', ('Domenie', 'Lunis', 'Martars', 'Miarcus', 'Joibe', 'Vinars', 'Sabide')),
   ('abmon', ('Zen', 'Fev', 'Mar', 'Avr', 'Mai', 'Jug', 'Lui', 'Avo', 'Set', 'Otu', 'Nov', 'Dec')),
   ('mon', ('Zenâr', 'Fevrâr', 'Març', 'Avrîl', 'Mai', 'Jugn', 'Lui', 'Avost', 'Setembar', 'Otubar', 'Novembar', 'Decembar')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d. %m. %y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSjJoOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('fy_DE',
  [('abday', ('Sdg', 'Mdg', 'Dsg', 'Mwk', 'Ddg', 'Fdg', 'Swd')),
   ('day', ('Sinndag', 'Mondag', 'Dingsdag', 'Meddwäakj', 'Donnadag', 'Friedag', 'Sinnowend')),
   ('abmon', ('Jan', 'Feb', 'Moz', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Now', 'Dez')),
   ('mon', ('Jaunuwoa', 'Februwoa', 'Moaz', 'Aprell', 'Mai', 'Juni', 'Juli', 'August', 'Septamba', 'Oktoba', 'Nowamba', 'Dezamba')),
   ('d_t_fmt', '%a %d. %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('fy_NL',
  [('abday', ('Sn', 'Mo', 'Ti', 'Wo', 'To', 'Fr', 'Sn')),
   ('day', ('Snein', 'Moandei', 'Tiisdei', 'Woansdei', 'Tongersdei', 'Freed', 'Sneon')),
   ('abmon', ('Jan', 'Feb', 'Maa', 'Apr', 'Maa', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Des')),
   ('mon', ('Janaris', 'Febrewaris', 'Maart', 'April', 'Maaie', 'Juny', 'July', 'Augustus', 'Septimber', 'Oktober', 'Novimber', 'Desimber')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ga_IE',
  [('abday', ('Domh', 'Luan', 'Máirt', 'Céad', 'Déar', 'Aoine', 'Sath')),
   ('day', ('Dé Domhnaigh', 'Dé Luain', 'Dé Máirt', 'Dé Céadaoin', 'Déardaoin', 'Dé hAoine', 'Dé Sathairn')),
   ('abmon', ('Ean', 'Feabh', 'Márta', 'Aib', 'Beal', 'Meith', 'Iúil', 'Lún', 'MFómh', 'DFómh', 'Samh', 'Noll')),
   ('mon', ('Eanáir', 'Feabhra', 'Márta', 'Aibreán', 'Bealtaine', 'Meitheamh', 'Iúil', 'Lúnasa', 'Meán Fómhair', 'Deireadh Fómhair', 'Samhain', 'Nollaig')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[tTyY].*'),
   ('noexpr', '^[nN].*')]),
 ('gd_GB',
  [('abday', ('DiD', 'DiL', 'DiM', 'DiC', 'Dia', 'Dih', 'DiS')),
   ('day', ('DiDòmhnaich', 'DiLuain', 'DiMàirt', 'DiCiadain', 'DiarDaoin', 'DihAoine', 'DiSathairne')),
   ('abmon', ('Faoi', 'Gearr', 'Màrt', 'Gibl', 'Mhàrt', 'Ògmh', 'Iuch', 'Lùna', 'Sult', 'Dàmh', 'Samh', 'Dùbh')),
   ('mon',
    ('Am Faoilleach',
     'An Gearran',
     'Am Màrt',
     'An Giblean',
     'An Cèitean',
     'An t-Ògmhios',
     'An t-Iuchar',
     'An Lùnastal',
     'An t-Sultain',
     'An Dàmhair',
     'An t-Samhain',
     'An Dùbhlachd')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[tTyY].*'),
   ('noexpr', '^[cCnN].*')]),
 ('gez_ER',
  [('abday', ('እኁድ', 'ሰኑይ', 'ሠሉስ', 'ራብዕ', 'ሐሙስ', 'ዓርበ', 'ቀዳሚ')),
   ('day', ('እኁድ', 'ሰኑይ', 'ሠሉስ', 'ራብዕ', 'ሐሙስ', 'ዓርበ', 'ቀዳሚት')),
   ('abmon', ('ጠሐረ', 'ከተተ', 'መገበ', 'አኀዘ', 'ግንባ', 'ሠንየ', 'ሐመለ', 'ነሐሰ', 'ከረመ', 'ጠቀመ', 'ኀደረ', 'ኀሠሠ')),
   ('mon', ('ጠሐረ', 'ከተተ', 'መገበ', 'አኀዘ', 'ግንባት', 'ሠንየ', 'ሐመለ', 'ነሐሰ', 'ከረመ', 'ጠቀመ', 'ኀደረ', 'ኀሠሠ')),
   ('d_t_fmt', '%A፥%B፡%e፡መዓልት፡%Y፡%r፡%Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X፡%p'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('gez_ET',
  [('abday', ('እኁድ', 'ሰኑይ', 'ሠሉስ', 'ራብዕ', 'ሐሙስ', 'ዓርበ', 'ቀዳሚ')),
   ('day', ('እኁድ', 'ሰኑይ', 'ሠሉስ', 'ራብዕ', 'ሐሙስ', 'ዓርበ', 'ቀዳሚት')),
   ('abmon', ('ጃንዩ', 'ፌብሩ', 'ማርች', 'ኤፕረ', 'ሜይ ', 'ጁን ', 'ጁላይ', 'ኦገስ', 'ሴፕቴ', 'ኦክተ', 'ኖቬም', 'ዲሴም')),
   ('mon', ('ጃንዩወሪ', 'ፌብሩወሪ', 'ማርች', 'ኤፕረል', 'ሜይ', 'ጁን', 'ጁላይ', 'ኦገስት', 'ሴፕቴምበር', 'ኦክተውበር', 'ኖቬምበር', 'ዲሴምበር')),
   ('d_t_fmt', '%A፥%B፡%e፡መዓልት፡%Y፡%r፡%Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X፡%p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('gl_ES',
  [('abday', ('Dom', 'Lun', 'Mar', 'Mér', 'Xov', 'Ven', 'Sáb')),
   ('day', ('Domingo', 'Luns', 'Martes', 'Mércores', 'Xoves', 'Venres', 'Sábado')),
   ('abmon', ('Xan', 'Feb', 'Mar', 'Abr', 'Mai', 'Xuñ', 'Xul', 'Ago', 'Set', 'Out', 'Nov', 'Dec')),
   ('mon', ('Xaneiro', 'Febreiro', 'Marzo', 'Abril', 'Maio', 'Xuño', 'Xullo', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Decembro')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('gu_IN',
  [('abday', ('રવિ', 'સોમ', 'મંગળ', 'બુધ', 'ગુરુ', 'શુક્ર', 'શનિ')),
   ('day', ('રવિવાર', 'સોમવાર', 'મંગળવાર', 'બુધવાર', 'ગુરુવાર', 'શુક્રવાર', 'શનિવાર')),
   ('abmon', ('જાન', 'ફેબ', 'માર', 'એપ્ર', 'મે', 'જુન', 'જુલ', 'ઓગ', 'સપ્ટ', 'ઓક્ટ', 'નોવ', 'ડિસ')),
   ('mon', ('જાન્યુઆરી', 'ફેબ્રુઆરી', 'માર્ચ', 'એપ્રિલ', 'મે', 'જુન', 'જુલાઇ', 'ઓગસ્ટ', 'સપ્ટેમ્બર', 'ઓક્ટોબર', 'નવેમ્બર', 'ડિસેમ્બર')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYહ]'),
   ('noexpr', '^[nNન]')]),
 ('gv_GB',
  [('abday', ('Jed', 'Jel', 'Jem', 'Jerc', 'Jerd', 'Jeh', 'Jes')),
   ('day', ('Jedoonee', 'Jelhein', 'Jemayrt', 'Jercean', 'Jerdein', 'Jeheiney', 'Jesarn')),
   ('abmon', ('J-guer', 'T-arree', 'Mayrnt', 'Avrril', 'Boaldyn', 'M-souree', 'J-souree', 'Luanistyn', 'M-fouyir', 'J-fouyir', 'M.Houney', 'M.Nollick')),
   ('mon',
    ('Jerrey-geuree',
     'Toshiaght-arree',
     'Mayrnt',
     'Averil',
     'Boaldyn',
     'Mean-souree',
     'Jerrey-souree',
     'Luanistyn',
     'Mean-fouyir',
     'Jerrey-fouyir',
     'Mee Houney',
     'Mee ny Nollick')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('ha_NG',
  [('abday', ('Lah', 'Lit', 'Tal', 'Lar', 'Alh', 'Jum', 'Asa')),
   ('day', ('Lahadi', 'Litini', 'Talata', 'Laraba', 'Alhamis', "Juma'a", 'Asabar')),
   ('abmon', ('Jan', 'Fab', 'Mar', 'Afr', 'May', 'Yun', 'Yul', 'Agu', 'Sat', 'Okt', 'Nuw', 'Dis')),
   ('mon', ('Janairu', 'Fabrairu', 'Maris', 'Afrilu', 'Mayu', 'Yuni', 'Yuli', 'Agusta', 'Satumba', 'Oktoba', 'Nuwamba', 'Disamba')),
   ('d_t_fmt', 'ranar %A, %d ga %B cikin %r %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[TtiIYy].*'),
   ('noexpr', '^[bBaAnN].*')]),
 ('hak_TW',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('禮拜日', '禮拜一', '禮拜二', '禮拜三', '禮拜四', '禮拜五', '禮拜六')),
   ('abmon', (' 1月', ' 2月', ' 3月', ' 4月', ' 5月', ' 6月', ' 7月', ' 8月', ' 9月', '10月', '11月', '12月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%Y年%m月%d日 (%A) %H點%M分%S秒'),
   ('d_fmt', '%Y年%m月%d日'),
   ('t_fmt', '%H點%M分%S秒'),
   ('t_fmt_ampm', '%p %I點%M分%S秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY係]'),
   ('noexpr', '^[nN毋]')]),
 ('he_IL',
  [('abday', ("א'", "ב'", "ג'", "ד'", "ה'", "ו'", "ש'")),
   ('day', ('ראשון', 'שני', 'שלישי', 'רביעי', 'חמישי', 'שישי', 'שבת')),
   ('abmon', ('ינו', 'פבר', 'מרץ', 'אפר', 'מאי', 'יונ', 'יול', 'אוג', 'ספט', 'אוק', 'נוב', 'דצמ')),
   ('mon', ('ינואר', 'פברואר', 'מרץ', 'אפריל', 'מאי', 'יוני', 'יולי', 'אוגוסט', 'ספטמבר', 'אוקטובר', 'נובמבר', 'דצמבר')),
   ('d_t_fmt', '%Z %H:%M:%S %Y %b %d %a'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S %P'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[Yyכ].*'),
   ('noexpr', '^[Nnל].*')]),
 ('hi_IN',
  [('abday', ('रवि ', 'सोम ', 'मंगल ', 'बुध ', 'गुरु ', 'शुक्र ', 'शनि ')),
   ('day', ('रविवार ', 'सोमवार ', 'मंगलवार ', 'बुधवार ', 'गुरुवार ', 'शुक्रवार ', 'शनिवार ')),
   ('abmon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('hne_IN',
  [('abday', ('इत ', 'सोम ', 'मंग ', 'बुध ', 'बिर ', 'सुक', 'सनि')),
   ('day', ('इतवार ', 'सोमवार ', 'मंगलवार ', 'बुधवार ', 'बिरसपत ', 'सुकरवार ', 'सनिवार ')),
   ('abmon', ('जन', 'फर', 'मार्च', 'अप', 'मई', 'जून', 'जुला', 'अग', 'सित', 'अकटू', 'नव', 'दिस')),
   ('mon', ('जनवरी', 'फरवरी', 'मार्च', 'अपरेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितमबर', 'अकटूबर', 'नवमबर', 'दिसमबर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[हवyY]'),
   ('noexpr', '^[नइnN]')]),
 ('hr_HR',
  [('abday', ('Ned', 'Pon', 'Uto', 'Sri', 'Čet', 'Pet', 'Sub')),
   ('day', ('Nedjelja', 'Ponedjeljak', 'Utorak', 'Srijeda', 'Četvrtak', 'Petak', 'Subota')),
   ('abmon', ('Sij', 'Vel', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro')),
   ('mon', ('Siječanj', 'Veljača', 'Ožujak', 'Travanj', 'Svibanj', 'Lipanj', 'Srpanj', 'Kolovoz', 'Rujan', 'Listopad', 'Studeni', 'Prosinac')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[dDyY].*'),
   ('noexpr', '^[nN].*')]),
 ('hsb_DE',
  [('abday', ('Nj', 'Pó', 'Wu', 'Sr', 'Št', 'Pj', 'So')),
   ('day', ('Njedźela', 'Póndźela', 'Wutora', 'Srjeda', 'Štvórtk', 'Pjatk', 'Sobota')),
   ('abmon', ('Jan', 'Feb', 'Měr', 'Apr', 'Mej', 'Jun', 'Jul', 'Awg', 'Sep', 'Okt', 'Now', 'Dec')),
   ('mon', ('Januar', 'Februar', 'Měrc', 'Apryl', 'Meja', 'Junij', 'Julij', 'Awgust', 'September', 'Oktober', 'Nowember', 'December')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[hHyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ht_HT',
  [('abday', ('dim', 'len ', 'mad', 'mèk', 'jed', 'van', 'sam')),
   ('day', ('dimanch', 'lendi ', 'madi', 'mèkredi', 'jedi', 'vandredi', 'samdi')),
   ('abmon', ('jan', 'fev', 'mas', 'avr', 'me', 'jen', 'jiy', 'out', 'sep', 'okt', 'nov', 'des')),
   ('mon', ('janvye', 'fevriye', 'mas', 'avril', 'me', 'jen', 'jiyè', 'out', 'septanm', 'oktòb', 'novanm', 'desanm')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[wWoOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('hu_HU',
  [('abday', ('v', 'h', 'k', 'sze', 'cs', 'p', 'szo')),
   ('day', ('vasárnap', 'hétfő', 'kedd', 'szerda', 'csütörtök', 'péntek', 'szombat')),
   ('abmon', ('jan', 'febr', 'márc', 'ápr', 'máj', 'jún', 'júl', 'aug', 'szept', 'okt', 'nov', 'dec')),
   ('mon', ('január', 'február', 'március', 'április', 'május', 'június', 'július', 'augusztus', 'szeptember', 'október', 'november', 'december')),
   ('d_t_fmt', '%Y. %b. %e., %A, %H.%M.%S %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%H.%M.%S'),
   ('t_fmt_ampm', '%H.%M.%S'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[IiYy].*'),
   ('noexpr', '^[nN].*')]),
 ('hy_AM',
  [('abday', ('Կրկ', 'Երկ', 'Երք', 'Չրք', 'Հնգ', 'Ուր', 'Շբթ')),
   ('day', ('Կիրակի', 'Երկուշաբթի', 'Երեքշաբթի', 'Չորեքշաբթի', 'Հինգշաբթի', 'Ուրբաթ', 'Շաբաթ')),
   ('abmon', ('Հնվ', 'Փտր', 'Մար', 'Ապր', 'Մայ', 'Հնս', 'Հլս', 'Օգս', 'Սեպ', 'Հոկ', 'Նմբ', 'Դեկ')),
   ('mon', ('Հունվարի', 'Փետրվարի', 'Մարտի', 'Ապրիլի', 'Մայիսի', 'Հունիսի', 'Հուլիսի', 'Օգոստոսի', 'Սեպտեմբերի', 'Հոկտեմբերի', 'Նոյեմբերի', 'Դեկտեմբերի')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%m/%d/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYsSաԱ]'),
   ('noexpr', '^[nNոՈ]')]),
 ('ia_FR',
  [('abday', ('dom', 'lun', 'mar', 'mer', 'jov', 'ven', 'sab')),
   ('day', ('dominica', 'lunedi', 'martedi', 'mercuridi', 'jovedi', 'venerdi', 'sabbato')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'mai', 'jun', 'jul', 'aug', 'sep', 'oct', 'nov', 'dec')),
   ('mon', ('januario', 'februario', 'martio', 'april', 'maio', 'junio', 'julio', 'augusto', 'septembre', 'octobre', 'novembre', 'decembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('id_ID',
  [('abday', ('Min', 'Sen', 'Sel', 'Rab', 'Kam', 'Jum', 'Sab')),
   ('day', ('Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu')),
   ('abmon', ('Jan', 'Peb', 'Mar', 'Apr', 'Mei', 'Jun', 'Jul', 'Agu', 'Sep', 'Okt', 'Nov', 'Des')),
   ('mon', ('Januari', 'Pebruari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[tTnN].*')]),
 ('ig_NG',
  [('abday', ('sọn', 'mọn', 'tuz', 'wen', 'tọs', 'fra', 'sat')),
   ('day', ('sọnde', 'mọnde', 'tuzde', 'wenzde', 'tọsde', 'fraịde', 'satọde')),
   ('abmon', ('jen', 'feb', 'maa', 'epr', 'mee', 'juu', 'jul', 'ọgọ', 'sep', 'ọkt', 'nọv', 'dis')),
   ('mon', ('jenụwarị', 'febụrụwarị', 'maachị', 'epreel', 'mee', 'juun', 'julaị', 'ọgọstụ', 'septemba', 'ọktoba', 'nọvemba', 'disemba')),
   ('d_t_fmt', '%A, %d %B %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[EeIiYy].*'),
   ('noexpr', '^[0MmNn].*')]),
 ('ik_CA',
  [('abday', ('Min', 'Sav', 'Ila', 'Qit', 'Sis', 'Tal', 'Maq')),
   ('day', ('Minġuiqsioiq', 'Savałłiq', 'Ilaqtchiioiq', 'Qitchiioiq', 'Sisamiioiq', 'Tallimmiioiq', 'Maqinġuoiq')),
   ('abmon', ('Sñt', 'Sñs', 'Pan', 'Qil', 'Sup', 'Iġñ', 'Itc', 'Tiñ', 'Ami', 'Sik', 'Nip', 'Siq')),
   ('mon',
    ('Siqiññaatchiaq',
     'Siqiññaasrugruk',
     'Paniqsiqsiivik',
     'Qilġich Tatqiat',
     'Suppivik',
     'Iġñivik',
     'Itchavik',
     'Tiññivik',
     'Amiġaiqsivik',
     'Sikkuvik',
     'Nippivik',
     'Siqiñġiḷaq')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '[yYiIaA].*'),
   ('noexpr', '[nNqQ].*')]),
 ('is_IS',
  [('abday', ('sun', 'mán', 'þri', 'mið', 'fim', 'fös', 'lau')),
   ('day', ('sunnudagur', 'mánudagur', 'þriðjudagur', 'miðvikudagur', 'fimmtudagur', 'föstudagur', 'laugardagur')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'maí', 'jún', 'júl', 'ágú', 'sep', 'okt', 'nóv', 'des')),
   ('mon', ('janúar', 'febrúar', 'mars', 'apríl', 'maí', 'júní', 'júlí', 'ágúst', 'september', 'október', 'nóvember', 'desember')),
   ('d_t_fmt', '%a %e.%b %Y, %T %Z'),
   ('d_fmt', '%a %e.%b %Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('it_CH',
  [('abday', ('dom', 'lun', 'mar', 'mer', 'gio', 'ven', 'sab')),
   ('day', ('domenica', 'lunedì', 'martedì', 'mercoledì', 'giovedì', 'venerdì', 'sabato')),
   ('abmon', ('gen', 'feb', 'mar', 'apr', 'mag', 'giu', 'lug', 'ago', 'set', 'ott', 'nov', 'dic')),
   ('mon', ('gennaio', 'febbraio', 'marzo', 'aprile', 'maggio', 'giugno', 'luglio', 'agosto', 'settembre', 'ottobre', 'novembre', 'dicembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d. %m. %y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', "'"),
   ('yesexpr', '^[sSjJoOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('it_IT',
  [('abday', ('dom', 'lun', 'mar', 'mer', 'gio', 'ven', 'sab')),
   ('day', ('domenica', 'lunedì', 'martedì', 'mercoledì', 'giovedì', 'venerdì', 'sabato')),
   ('abmon', ('gen', 'feb', 'mar', 'apr', 'mag', 'giu', 'lug', 'ago', 'set', 'ott', 'nov', 'dic')),
   ('mon', ('gennaio', 'febbraio', 'marzo', 'aprile', 'maggio', 'giugno', 'luglio', 'agosto', 'settembre', 'ottobre', 'novembre', 'dicembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('iu_CA',
  [('abday', ('ᓈ', 'ᓇ', 'ᓕ', 'ᐱ', 'ᕿ', 'ᐅ', 'ᓯ')),
   ('day', ('ᓈᑦᑎᖑᔭᕐᕕᒃ', 'ᓇᒡᒐᔾᔭᐅ', 'ᓇᒡᒐᔾᔭᐅᓕᖅᑭᑦ', 'ᐱᖓᓲᓕᖅᓯᐅᑦ', 'ᕿᑎᖅᑰᑦ', 'ᐅᓪᓗᕈᓘᑐᐃᓇᖅ', 'ᓯᕙᑖᕕᒃ')),
   ('abmon', ('ᔮᓄ', 'ᕕᕗ', 'ᒪᔅ', 'ᐃᐳ', 'ᒪᐃ', 'ᔪᓂ', 'ᔪᓚ', 'ᐊᒋ', 'ᓯᑎ', 'ᐊᑦ', 'ᓄᕕ', 'ᑎᓯ')),
   ('mon', ('ᔮᓄᐊᓕ', 'ᕕᕗᐊᓕ', 'ᒪᔅᓯ', 'ᐃᐳᓗ', 'ᒪᐃ', 'ᔪᓂ', 'ᔪᓚᐃ', 'ᐊᒋᓯ', 'ᓯᑎᕙ', 'ᐊᑦᑐᕙ', 'ᓄᕕᕙ', 'ᑎᓯᕝᕙ')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%m/%d/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '[yYsS].*'),
   ('noexpr', '[nN].*')]),
 ('iw_IL',
  [('abday', ("א'", "ב'", "ג'", "ד'", "ה'", "ו'", "ש'")),
   ('day', ('ראשון', 'שני', 'שלישי', 'רביעי', 'חמישי', 'שישי', 'שבת')),
   ('abmon', ('ינו', 'פבר', 'מרץ', 'אפר', 'מאי', 'יונ', 'יול', 'אוג', 'ספט', 'אוק', 'נוב', 'דצמ')),
   ('mon', ('ינואר', 'פברואר', 'מרץ', 'אפריל', 'מאי', 'יוני', 'יולי', 'אוגוסט', 'ספטמבר', 'אוקטובר', 'נובמבר', 'דצמבר')),
   ('d_t_fmt', '%Z %H:%M:%S %Y %b %d %a'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S %P'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[Yyכ].*'),
   ('noexpr', '^[Nnל].*')]),
 ('ja_JP',
  [('abday', ('日', '月', '火', '水', '木', '金', '土')),
   ('day', ('日曜日', '月曜日', '火曜日', '水曜日', '木曜日', '金曜日', '土曜日')),
   ('abmon', (' 1月', ' 2月', ' 3月', ' 4月', ' 5月', ' 6月', ' 7月', ' 8月', ' 9月', '10月', '11月', '12月')),
   ('mon', ('1月', '2月', '3月', '4月', '5月', '6月', '7月', '8月', '9月', '10月', '11月', '12月')),
   ('d_t_fmt', '%Y年%m月%d日 %H時%M分%S秒'),
   ('d_fmt', '%Y年%m月%d日'),
   ('t_fmt', '%H時%M分%S秒'),
   ('t_fmt_ampm', '%p%I時%M分%S秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^([yYｙＹ]|はい|ハイ)'),
   ('noexpr', '^([nNｎＮ]|いいえ|イイエ)')]),
 ('ka_GE',
  [('abday', ('კვი', 'ორშ', 'სამ', 'ოთხ', 'ხუთ', 'პარ', 'შაბ')),
   ('day', ('კვირა', 'ორშაბათი', 'სამშაბათი', 'ოთხშაბათი', 'ხუთშაბათი', 'პარასკევი', 'შაბათი')),
   ('abmon', ('იან', 'თებ', 'მარ', 'აპრ', 'მაი', 'ივნ', 'ივლ', 'აგვ', 'სექ', 'ოქტ', 'ნოე', 'დეკ')),
   ('mon', ('იანვარი', 'თებერვალი', 'მარტი', 'აპრილი', 'მაისი', 'ივნისი', 'ივლისი', 'აგვისტო', 'სექტემბერი', 'ოქტომბერი', 'ნოემბერი', 'დეკემბერი')),
   ('d_t_fmt', '%Y წლის %d %B, %T %Z'),
   ('d_fmt', '%m/%d/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[1yYkKxXხ].*'),
   ('noexpr', '^[0nNaAა].*')]),
 ('kk_KZ',
  [('abday', ('Жк', 'Дс', 'Сс', 'Ср', 'Бс', 'Жм', 'Сн')),
   ('day', ('Жексенбі', 'Дүйсенбі', 'Сейсенбі', 'Сәрсенбі', 'Бейсенбі', 'Жұма', 'Сенбі')),
   ('abmon', ('Қаң', 'Ақп', 'Нау', 'Сәу', 'Мам', 'Мау', 'Шіл', 'Там', 'Қыр', 'Қаз', 'Қар', 'Жел')),
   ('mon', ('Қаңтар', 'Ақпан', 'Наурыз', 'Сәуір', 'Мамыр', 'Маусым', 'Шілде', 'Тамыз', 'Қыркүйек', 'Қазан', 'Қараша', 'Желтоқсан')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[ИиYy].*'),
   ('noexpr', '^[ЖжNn].*')]),
 ('kl_GL',
  [('abday', ('sab', 'ata', 'mar', 'pin', 'sis', 'tal', 'arf')),
   ('day', ('sabaat', 'ataasinngorneq', 'marlunngorneq', 'pingasunngorneq', 'sisamanngorneq', 'tallimanngorneq', 'arfininngorneq')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'maj', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januari', 'februari', 'martsi', 'aprili', 'maji', 'juni', 'juli', 'augustusi', 'septemberi', 'oktoberi', 'novemberi', 'decemberi')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d %b %Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[JjYyAa].*'),
   ('noexpr', '^[Nn].*')]),
 ('km_KH',
  [('abday', ('អា', 'ច', 'អ', 'ពុ', 'ព្រ', 'សុ', 'ស')),
   ('day', ('ថ្ងៃ\u200bអាទិត្យ', 'ថ្ងៃ\u200bច័ន្ទ', 'ថ្ងៃ\u200bអង្គារ', 'ថ្ងៃ\u200bពុធ', 'ថ្ងៃ\u200bព្រហស្បតិ៍', 'ថ្ងៃ\u200bសុក្រ', 'ថ្ងៃ\u200bសៅរ៍')),
   ('abmon', ('១', '២', '៣', '៤', '៥', '៦', '៧', '៨', '៩', '១០', '១១', '១២')),
   ('mon', ('មករា', 'កុម្ភៈ', 'មិនា', 'មេសា', 'ឧសភា', 'មិថុនា', 'កក្កដា', 'សីហា', 'កញ្ញា', 'តុលា', 'វិច្ឆិកា', 'ធ្នូ')),
   ('d_t_fmt', '%A ថ្ងៃ %e ខែ %B ឆ្នាំ  %Y, %H ម៉ោង m នាទី %S វិនាទី\u200b'),
   ('d_fmt', '%e %B %Y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]([eE][sS])?'),
   ('noexpr', '^[nN][oO]?')]),
 ('kn_IN',
  [('abday', ('ರ', 'ಸೋ', 'ಮಂ', 'ಬು', 'ಗು', 'ಶು', 'ಶ')),
   ('day', ('ರವಿವಾರ', 'ಸೋಮವಾರ', 'ಮಂಗಳವಾರ', 'ಬುಧವಾರ', 'ಗುರುವಾರ', 'ಶುಕ್ರವಾರ', 'ಶನಿವಾರ')),
   ('abmon', ('ಜ', 'ಫೆ', 'ಮಾ', 'ಏ', 'ಮೇ', 'ಜೂ', 'ಜು', 'ಆ', 'ಸೆ', 'ಅ', 'ನ', 'ದ')),
   ('mon', ('ಜನವರಿ', 'ಫೆಬ್ರವರಿ', 'ಮಾರ್ಚ', 'ಏಪ್ರಿಲ್', 'ಮೇ', 'ಜೂನ್', 'ಜುಲಾಯಿ', 'ಆಗಸ್ತು', 'ಸೆಪ್ಟೆಂಬರ', 'ಅಕ್ತೂಬರ', 'ನವೆಂಬರ', 'ದಶಂಬರ')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('ko_KR',
  [('abday', ('일', '월', '화', '수', '목', '금', '토')),
   ('day', ('일요일', '월요일', '화요일', '수요일', '목요일', '금요일', '토요일')),
   ('abmon', (' 1월', ' 2월', ' 3월', ' 4월', ' 5월', ' 6월', ' 7월', ' 8월', ' 9월', '10월', '11월', '12월')),
   ('mon', ('1월', '2월', '3월', '4월', '5월', '6월', '7월', '8월', '9월', '10월', '11월', '12월')),
   ('d_t_fmt', '%x (%a) %r'),
   ('d_fmt', '%Y년 %m월 %d일'),
   ('t_fmt', '%H시 %M분 %S초'),
   ('t_fmt_ampm', '%p %I시 %M분 %S초'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY예]'),
   ('noexpr', '^[nN아]')]),
 ('kok_IN',
  [('abday', ('आयतार', 'सोमार', 'मंगळवार', 'बुधवार', 'बेरेसतार', 'शुकरार', 'शेनवार')),
   ('day', ('आयतार', 'सोमार', 'मंगळवार', 'बुधवार', 'बेरेसतार', 'शुकरार', 'शेनवार')),
   ('abmon', ('जेनवरी', 'फेब्ररी', 'मारच', 'एप्रील', 'में', 'जुन', 'जुलै', 'ओगस्ट', 'सेपटेंबर', 'ओकटोबर', 'नोवेंबर', 'दिसेंबर')),
   ('mon', ('जेनवरी', 'फेब्ररी', 'मारच', 'एप्रील', 'में', 'जुन', 'जुलै', 'ओगस्ट', 'सेपटेंबर', 'ओकटोबर', 'नोवेंबर', 'दिसेंबर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^(हय|[yY])'),
   ('noexpr', '^(न्ही|[nN])')]),
 ('ks_IN',
  [('abday', ('آتهوار', 'ژءنتروار', 'بوءںوار', 'بودهوار', 'برىسوار', 'جمع', 'بٹوار')),
   ('day', ('آتهوار', 'ژءندروار', 'بوءںوار', 'بودهوار', 'برىسوار', 'جمع', 'بٹوار')),
   ('abmon', ('جنوری', 'فروری', 'مارچ', 'اپریل', 'مئ', 'جون', 'جُلئ', 'اگست', 'ستنبر', 'اکتوبر', 'نوںبر', 'دسنبر')),
   ('mon', ('جنوری', 'فروری', 'مارچ', 'اپریل', 'مئ', 'جون', 'جُلئ', 'اگست', 'ستنبر', 'اکتوبر', 'نوںبر', 'دسنبر')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[آyY].*'),
   ('noexpr', '^[نnN].*')]),
 ('ku_TR',
  [('abday', ('yêk', 'dus', 'sês', 'çar', 'pên', 'înî', 'sep')),
   ('day', ('yêksêm', 'dusêm', 'sêsêm', 'çarsêm', 'pêncsêm', 'înî', 'sept')),
   ('abmon', ('Çil', 'Sib', 'Ada', 'Nîs', 'Gul', 'Hez', 'Tîr', 'Teb', 'Îlo', 'Cot', 'Mij', 'Kan')),
   ('mon', ('Çile', 'Sibat', 'Adar', 'Nîsan', 'Gulan', 'Hezîran', 'Tîrmeh', 'Tebax', 'Îlon', 'Cotmeh', 'Mijdar', 'Kanûn')),
   ('d_t_fmt', '%A %d %B %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[eEdDyY].*'),
   ('noexpr', '^[nN].*')]),
 ('kw_GB',
  [('abday', ('Sul', 'Lun', 'Mth', 'Mhr', 'Yow', 'Gwe', 'Sad')),
   ('day', ('De Sul', 'De Lun', 'De Merth', 'De Merher', 'De Yow', 'De Gwener', 'De Sadorn')),
   ('abmon', ('Gen', 'Whe>', 'Mer', 'Ebr', 'Me', 'Evn', 'Gor', 'Est', 'Gwn', 'Hed', 'Du', 'Kev')),
   ('mon',
    ('Mys Genver',
     'Mys Whevrel',
     'Mys Merth',
     'Mys Ebrel',
     'Mys Me',
     'Mys Evan',
     'Mys Gortheren',
     'Mye Est',
     'Mys Gwyngala',
     'Mys Hedra',
     'Mys Du',
     'Mys Kevardhu')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[eEyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ky_KG',
  [('abday', ('жк', 'дш', 'ше', 'ша', 'бш', 'жм', 'иш')),
   ('day', ('жекшемби', 'дүйшөмбү', 'шейшемби', 'шаршемби', 'бейшемби', 'жума', 'ишемби')),
   ('abmon', ('янв', 'фев', 'мар', 'апр', 'май', 'июн', 'июл', 'авг', 'сен', 'окт', 'ноя', 'дек')),
   ('mon', ('январь', 'февраль', 'март', 'апрель', 'май', 'июнь', 'июль', 'август', 'сентябрь', 'октябрь', 'ноябрь', 'декабрь')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[ОоYy].*'),
   ('noexpr', '^[ЖжNn].*')]),
 ('lb_LU',
  [('abday', ('So', 'Mé', 'Dë', 'Më', 'Do', 'Fr', 'Sa')),
   ('day', ('Sonndeg', 'Méindeg', 'Dënschdeg', 'Mëttwoch', 'Donneschdeg', 'Freideg', 'Samschdeg')),
   ('abmon', ('Jan', 'Feb', 'Mäe', 'Abr', 'Mee', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dez')),
   ('mon', ('Januar', 'Februar', 'Mäerz', 'Abrëll', 'Mee', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember')),
   ('d_t_fmt', '%a %d. %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('lg_UG',
  [('abday', ('Sab', 'Bal', 'Lw2', 'Lw3', 'Lw4', 'Lw5', 'Lw6')),
   ('day', ('Sabiiti', 'Balaza', 'Lwakubiri', 'Lwakusatu', 'Lwakuna', 'Lwakutaano', 'Lwamukaaga')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apu', 'Maa', 'Jun', 'Jul', 'Agu', 'Seb', 'Oki', 'Nov', 'Des')),
   ('mon', ('Janwaliyo', 'Febwaliyo', 'Marisi', 'Apuli', 'Maayi', 'Juuni', 'Julaai', 'Agusito', 'Sebuttemba', 'Okitobba', 'Novemba', 'Desemba')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('li_BE',
  [('abday', ('zón', 'mao', 'dae', 'goo', 'dón', 'vri', 'z\x91o')),
   ('day', ('zóndig', 'maondig', 'daensdig', 'goonsdig', 'dónderdig', 'vriedig', 'zaoterdig')),
   ('abmon', ('jan', 'fib', 'mie', 'epr', 'mei', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'des')),
   ('mon', ('jannewarie', 'fibberwarie', 'miert', 'eprèl', 'meij', 'junie', 'julie', 'augustus', 'september', 'oktober', 'november', 'desember')),
   ('d_t_fmt', '%a %d. %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('li_NL',
  [('abday', ('zón', 'mao', 'dae', 'goo', 'dón', 'vri', 'z\x91o')),
   ('day', ('zóndig', 'maondig', 'daensdig', 'goonsdig', 'dónderdig', 'vriedig', 'zaoterdig')),
   ('abmon', ('jan', 'fib', 'mie', 'epr', 'mei', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'des')),
   ('mon', ('jannewarie', 'fibberwarie', 'miert', 'eprèl', 'meij', 'junie', 'julie', 'augustus', 'september', 'oktober', 'november', 'desember')),
   ('d_t_fmt', '%a %d. %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('lij_IT',
  [('abday', ('dom', 'lûn', 'mar', 'mer', 'zêu', 'ven', 'sab')),
   ('day', ('domenega', 'lûnedì', 'martedì', 'mercUrdì', 'zêggia', 'venardì', 'sabbo')),
   ('abmon', ('zen', 'fev', 'mar', 'arv', 'maz', 'zûg', 'lûg', 'ago', 'set', 'ött', 'nov', 'dix')),
   ('mon', ('zenâ', 'fevrâ', 'marzo', 'avrî', 'mazzo', 'zûgno', 'lûggio', 'agosto', 'settembre', 'ottobre', 'novembre', 'dixembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('lo_LA',
  [('abday', ('ອາ.', 'ຈ.', 'ຄ.', 'ພ.', 'ພຫ.', 'ສ.', 'ສ.')),
   ('day', ('ອາທິດ', 'ຈັນ', 'ອັງຄານ', 'ພຸດ', 'ພະຫັດ', 'ສຸກ', 'ເສົາ')),
   ('abmon', ('ມ.ກ.', 'ກ.ພ.', 'ມ.ນ.', 'ມ.ສ.', 'ພ.ພ.', 'ມິ.ຖ.', 'ກ.ລ.', 'ສ.ຫ.', 'ກ.ຍ.', 'ຕ.ລ.', 'ພ.ຈ.', 'ທ.ວ.')),
   ('mon', ('ມັງກອນ', 'ກຸມຟາ', 'ມີນາ', 'ເມສາ', 'ພຶດສະພາ', 'ມິຖຸນາ', 'ກໍລະກົດ', 'ສິງຫາ', 'ກັນຍາ', 'ຕຸລາ', 'ພະຈິກ', 'ທັນວາ')),
   ('d_t_fmt', '%a %e %b %Ey, %H:%M:%S'),
   ('d_fmt', '%d/%m/%Ey'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYມ]'),
   ('noexpr', '^[nNບ]')]),
 ('lt_LT',
  [('abday', ('Sk', 'Pr', 'An', 'Tr', 'Kt', 'Pn', 'Št')),
   ('day', ('Sekmadienis', 'Pirmadienis', 'Antradienis', 'Trečiadienis', 'Ketvirtadienis', 'Penktadienis', 'Šeštadienis')),
   ('abmon', ('Sau', 'Vas', 'Kov', 'Bal', 'Geg', 'Bir', 'Lie', 'Rgp', 'Rgs', 'Spa', 'Lap', 'Grd')),
   ('mon', ('sausio', 'vasario', 'kovo', 'balandžio', 'gegužės', 'birželio', 'liepos', 'rugpjūčio', 'rugsėjo', 'spalio', 'lapkričio', 'gruodžio')),
   ('d_t_fmt', '%Y m. %B %d d. %T'),
   ('d_fmt', '%Y.%m.%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[TtYy].*'),
   ('noexpr', '^[Nn].*')]),
 ('lv_LV',
  [('abday', ('Sv', 'P\xa0', 'O\xa0', 'T\xa0', 'C\xa0', 'Pk', 'S\xa0')),
   ('day', ('svētdiena', 'pirmdiena', 'otrdiena', 'trešdiena', 'ceturtdiena', 'piektdiena', 'sestdiena')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'mai', 'jūn', 'jūl', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('janvāris', 'februāris', 'marts', 'aprīlis', 'maijs', 'jūnijs', 'jūlijs', 'augusts', 'septembris', 'oktobris', 'novembris', 'decembris')),
   ('d_t_fmt', '%A, %Y. gada %e. %B, plkst. %H un %M'),
   ('d_fmt', '%Y.%m.%d.'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[JjYy].*'),
   ('noexpr', '^[Nn].*')]),
 ('lzh_TW',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('週日', '週一', '週二', '週三', '週四', '週五', '週六')),
   ('abmon', (' 一 ', ' 二 ', ' 三 ', ' 四 ', ' 五 ', ' 六 ', ' 七 ', ' 八 ', ' 九 ', ' 十 ', '十一', '十二')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%OC%Oy年%B%Od日 (%A) %OH時%OM分%OS秒'),
   ('d_fmt', '%OC%Oy年%B%Od日'),
   ('t_fmt', '%OH時%OM分%OS秒'),
   ('t_fmt_ampm', '%p %OI時%OM分%OS秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY是]'),
   ('noexpr', '^[nN非]')]),
 ('mag_IN',
  [('abday', ('एतवार ', 'सोमार ', 'मंगर ', 'बुध ', 'बिफे ', 'सूक ', 'सनिचर ')),
   ('day', ('एतवार ', 'सोमार ', 'मंगर ', 'बुध ', 'बिफे ', 'सूक ', 'सनिचर ')),
   ('abmon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('mai_IN',
  [('abday', ('रवि ', 'सोम ', 'मंगल ', 'बुध ', 'गुरु ', 'शुक्र ', 'शनि ')),
   ('day', ('रविवार ', 'सोमवार ', 'मंगलवार ', 'बुधवार ', 'गुरुवार ', 'शुक्रवार ', 'शनिवार ')),
   ('abmon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('mg_MG',
  [('abday', ('lhd', 'lts', 'tlt', 'lrb', 'lkm', 'zom', 'sab')),
   ('day', ('alahady', 'alatsinainy', 'talata', 'alarobia', 'alakamisy', 'zoma', 'sabotsy')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'mey', 'jon', 'jol', 'aog', 'sep', 'okt', 'nov', 'des')),
   ('mon', ('janoary', 'febroary', 'martsa', 'aprily', 'mey', 'jona', 'jolay', 'aogositra', 'septambra', 'oktobra', 'novambra', 'desambra')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[eEyY].*'),
   ('noexpr', '^[tTnN].*')]),
 ('mhr_RU',
  [('abday', ('Ршр', 'Шчм', 'Кжм', 'Вгч', 'Изр', 'Кгр', 'Шмт')),
   ('day', ('Рушарня', 'Шочмо', 'Кушкыжмо', 'Вӱргече', 'Изарня', 'Кугарня', 'Шуматкече')),
   ('abmon', ('Шрк', 'Пгж', 'Ӱрн', 'Вшр', 'Ага', 'Пдш', 'Срм', 'Срл', 'Идм', 'Шыж', 'Клм', 'Тел')),
   ('mon', ('Шорыкйол', 'Пургыж', 'Ӱярня', 'Вӱдшор', 'Ага', 'Пеледыш', 'Сӱрем', 'Сорла', 'Идым', 'Шыжа', 'Кылме', 'Теле')),
   ('d_t_fmt', '%A %Y %B %d %T'),
   ('d_fmt', '%Y.%m.%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[ТтYy].*'),
   ('noexpr', '^[УуNn].*')]),
 ('mi_NZ',
  [('abday', ('Ta', 'Ma', 'Tū', 'We', 'Tāi', 'Pa', 'Hā')),
   ('day', ('Rātapu', 'Mane', 'Tūrei', 'Wenerei', 'Tāite', 'Paraire', 'Hātarei')),
   ('abmon', ('Kohi', 'Hui', 'Pou', 'Pae', 'Hara', 'Pipi', 'Hōngoi', 'Here', 'Mahu', 'Whi-nu', 'Whi-ra', 'Haki')),
   ('mon',
    ('Kohi-tātea',
     'Hui-tanguru',
     'Poutū-te-rangi',
     'Paenga-whāwhā',
     'Haratua',
     'Pipiri',
     'Hōngoingoi',
     'Here-turi-kōkā',
     'Mahuru',
     'Whiringa-ā-nuku',
     'Whiringa-ā-rangi',
     'Hakihea')),
   ('d_t_fmt', 'Te %A, te %d o %B, %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[1yYāĀäÄaA].*'),
   ('noexpr', '^[0nNkK].*')]),
 ('mk_MK',
  [('abday', ('нед', 'пон', 'вто', 'сре', 'чет', 'пет', 'саб')),
   ('day', ('недела', 'понеделник', 'вторник', 'среда', 'четврток', 'петок', 'сабота')),
   ('abmon', ('јан', 'фев', 'мар', 'апр', 'мај', 'јун', 'јул', 'авг', 'сеп', 'окт', 'ное', 'дек')),
   ('mon', ('јануари', 'февруари', 'март', 'април', 'мај', 'јуни', 'јули', 'август', 'септември', 'октомври', 'ноември', 'декември')),
   ('d_t_fmt', '%a, %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[ДдDdYy1].*'),
   ('noexpr', '^[НнNn0].*')]),
 ('ml_IN',
  [('abday', ('ഞാ', 'തി', 'ചൊ', 'ബു', 'വ്യാ', 'വെ', 'ശ')),
   ('day', ('ഞായര്\u200d', 'തിങ്കള്\u200d', 'ചൊവ്വ', 'ബുധന്\u200d', 'വ്യാഴം', 'വെള്ളി', 'ശനി')),
   ('abmon', ('ജനു', 'ഫെബ്', 'മാര്\u200d', 'ഏപ്ര', 'മെ', 'ജൂണ്\u200d', 'ജൂലൈ', 'ആഗ്', 'സെപ്', 'ഒക്ടോ', 'നവം', 'ഡിസം')),
   ('mon',
    ('ജനുവരി',
     'ഫെബ്രുവരി',
     'മാര്\u200dച്ച്',
     'ഏപ്രില്\u200d ',
     'മെയ്',
     'ജൂണ്\u200d',
     'ജൂലൈ',
     'ആഗസ്റ്റ്',
     'സെപ്റ്റംബര്\u200d',
     'ഒക്ടോബര്\u200d',
     'നവംബര്\u200d',
     'ഡിസംബര്\u200d')),
   ('d_t_fmt', '%A %d %B %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %B %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[അതെyY]'),
   ('noexpr', '^[അല്ലnN]')]),
 ('mn_MN',
  [('abday', ('Ня', 'Да', 'Мя', 'Лх', 'Пү', 'Ба', 'Бя')),
   ('day', ('Ням', 'Даваа', 'Мягмар', 'Лхагва', 'Пүрэв', 'Баасан', 'Бямба')),
   ('abmon', ('Хул', 'Үхэ', 'Бар', 'Туу', 'Луу', 'Мог', 'Мор', 'Хон', 'Бич', 'Тах', 'Нох', 'Гах')),
   ('mon',
    ('Хулгана сарын',
     'Үхэр сарын',
     'Бар сарын',
     'Туулай сарын',
     'Луу сарын',
     'Могой сарын',
     'Морь сарын',
     'Хонь сарын',
     'Бич сарын',
     'Тахиа сарын',
     'Нохой сарын',
     'Гахай сарын')),
   ('d_t_fmt', '%Y %b %d, %a %T'),
   ('d_fmt', '%Y.%m.%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[тТyY].*'),
   ('noexpr', '^[үҮnN].*')]),
 ('mni_IN',
  [('abday', ('নোং', 'নিং', 'লৈবাক', 'য়ুম', 'শগোল', 'ইরা', 'থাং')),
   ('day', ('নোংমাইজিং', 'নিংথৌকাবা', 'লৈবাকপোকপা', 'য়ুমশকৈশা', 'শগোলশেন', 'ইরাই', 'থাংজ')),
   ('abmon', ('জান', 'ফেব', 'মার', 'এপ্রি', 'মে', 'জুন', 'জুল', 'আগ', 'সেপ', 'ওক্ত', 'নবে', 'ডিস')),
   ('mon', ('জানুৱারি', 'ফেব্রুৱারি', 'মার্চ', 'এপ্রিল', 'মে', 'জুন', 'জুলাই', 'আগষ্ট', 'সেপ্তেম্বর', 'ওক্তোবর', 'নবেম্বর', 'ডিসেম্বর')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('mr_IN',
  [('abday', ('रवि', 'सोम', 'मंगळ', 'बुध', 'गुरु', 'शुक्र', 'शनि')),
   ('day', ('रविवार', 'सोमवार', 'मंगळवार', 'बुधवार', 'गुरुवार', 'शुक्रवार', 'शनिवार')),
   ('abmon', ('जानेवारी', 'फेबृवारी', 'मार्च', 'एप्रिल', 'मे', 'जून', 'जुलै', 'ऑगस्ट', 'सप्टेंबर', 'ऑक्टोबर', 'नोव्हेंबर', 'डिसेंबर')),
   ('mon', ('जानेवारी', 'फेबृवारी', 'मार्च', 'एप्रिल', 'मे', 'जून', 'जुलै', 'ऑगस्ट', 'सप्टेंबर', 'ऑक्टोबर', 'नोव्हेंबर', 'डिसेंबर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^(Yes|[yY])'),
   ('noexpr', '^(No|[nN])')]),
 ('ms_MY',
  [('abday', ('Ahd', 'Isn', 'Sel', 'Rab', 'Kha', 'Jum', 'Sab')),
   ('day', ('Ahad', 'Isnin', 'Selasa', 'Rabu', 'Khamis', 'Jumaat', 'Sabtu')),
   ('abmon', ('Jan', 'Feb', 'Mac', 'Apr', 'Mei', 'Jun', 'Jul', 'Ogos', 'Sep', 'Okt', 'Nov', 'Dis')),
   ('mon', ('Januari', 'Februari', 'Mac', 'April', 'Mei', 'Jun', 'Julai', 'Ogos', 'September', 'Oktober', 'November', 'Disember')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[tT]')]),
 ('mt_MT',
  [('abday', ('Ħad', 'Tne', 'Tli', 'Erb', 'Ħam', 'Ġim', 'Sib')),
   ('day', ('il-Ħadd', 'it-Tnejn', 'it-Tlieta', 'l-Erbgħa', 'il-Ħamis', 'il-Ġimgħa', 'is-Sibt')),
   ('abmon', ('Jan', 'Fra', 'Mar', 'Apr', 'Mej', 'Ġun', 'Lul', 'Aww', 'Set', 'Ott', 'Nov', 'Diċ')),
   ('mon', ('Jannar', 'Frar', 'Marzu', 'April', 'Mejju', 'Ġunju', 'Lulju', 'Awwissu', 'Settembru', 'Ottubru', 'Novembru', 'Diċembru ')),
   ('d_t_fmt', '%A, %d ta %b, %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A, %d ta %b, %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^(Yes|[yY])'),
   ('noexpr', '^(No|[nN])')]),
 ('my_MM',
  [('abday', ('နွေ', 'လာ', 'ဂါ', 'ဟူး', 'တေး', 'သော', 'နေ')),
   ('day', ('တနင်္ဂနွေ', 'တနင်္လာ', 'အင်္ဂါ', 'ဗုဒ္ဓဟူး', 'ကြာသပတေး', 'သောကြာ', 'စနေ')),
   ('abmon', ('ဇန်', 'ဖေ', 'မတ်', 'ဧပြီ', 'မေ', 'ဇွန်', 'ဇူ', 'ဩ', 'စက်', 'အောက်', 'နို', 'ဒီ')),
   ('mon', ('ဇန်နဝါရီ', 'ဖေဖော်ဝါရီ', 'မတ်', 'ဧပြီ', 'မေ', 'ဇွန်', 'ဇူလိုင်', 'ဩဂုတ်', 'စက်တင်ဘာ', 'အောက်တိုဘာ', 'နိုဝင်ဘာ', 'ဒီဇင်ဘာ')),
   ('d_t_fmt', '%OC%Oy %b %Od %A %OI:%OM:%OS %Op %Z'),
   ('d_fmt', '%OC%Oy %b %Od %A'),
   ('t_fmt', '%OI:%OM:%OS %p'),
   ('t_fmt_ampm', '%OI:%OM:%OS %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYဟ].*'),
   ('noexpr', '^[nNမ].*')]),
 ('nan_TW',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('禮拜日', '禮拜一', '禮拜二', '禮拜三', '禮拜四', '禮拜五', '禮拜六')),
   ('abmon', (' 1月', ' 2月', ' 3月', ' 4月', ' 5月', ' 6月', ' 7月', ' 8月', ' 9月', '10月', '11月', '12月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%Y年%m月%d日 (%A) %H點%M分%S秒'),
   ('d_fmt', '%Y年%m月%d日'),
   ('t_fmt', '%H點%M分%S秒'),
   ('t_fmt_ampm', '%p %I點%M分%S秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY是]'),
   ('noexpr', '^[nN伓]')]),
 ('nb_NO',
  [('abday', ('sø.', 'ma.', 'ti.', 'on.', 'to.', 'fr.', 'lø.')),
   ('day', ('søndag', 'mandag', 'tirsdag', 'onsdag', 'torsdag', 'fredag', 'lørdag')),
   ('abmon', ('jan.', 'feb.', 'mars', 'april', 'mai', 'juni', 'juli', 'aug.', 'sep.', 'okt.', 'nov.', 'des.')),
   ('mon', ('januar', 'februar', 'mars', 'april', 'mai', 'juni', 'juli', 'august', 'september', 'oktober', 'november', 'desember')),
   ('d_t_fmt', '%a %d. %b %Y kl. %H.%M %z'),
   ('d_fmt', '%d. %b %Y'),
   ('t_fmt', 'kl. %H.%M %z'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[JjYy].*'),
   ('noexpr', '^[Nn].*')]),
 ('nds_DE',
  [('abday', ('Sdag', 'Maan', 'Ding', 'Migg', 'Dunn', 'Free', 'Svd.')),
   ('day', ('Sünndag', 'Maandag', 'Dingsdag', 'Middeweek', 'Dunnersdag', 'Freedag', 'Sünnavend')),
   ('abmon', ('Jan', 'Feb', 'Mär', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Nov', 'Dez')),
   ('mon', ('Jannuaar', 'Feberwaar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember')),
   ('d_t_fmt', '%a %d. %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('nds_NL',
  [('abday', ('Sdg', 'Mdg', 'Dsg', 'Mwk', 'Ddg', 'Fdg', 'Swd')),
   ('day', ('Sinndag', 'Mondag', 'Dingsdag', 'Meddwäakj', 'Donnadag', 'Friedag', 'Sinnowend')),
   ('abmon', ('Jan', 'Feb', 'Moz', 'Apr', 'Mai', 'Jun', 'Jul', 'Aug', 'Sep', 'Okt', 'Now', 'Dez')),
   ('mon', ('Jaunuwoa', 'Februwoa', 'Moaz', 'Aprell', 'Mai', 'Juni', 'Juli', 'August', 'Septamba', 'Oktoba', 'Nowamba', 'Dezamba')),
   ('d_t_fmt', '%a %d. %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('ne_NP',
  [('abday', ('आइत ', 'सोम ', 'मंगल ', 'बुध ', 'बिहि ', 'शुक्र ', 'शनि ')),
   ('day', ('आइतबार ', 'सोमबार ', 'मंगलबार ', 'बुधबार ', 'बिहिबार ', 'शुक्रबार ', 'शनिबार ')),
   ('abmon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('nhn_MX',
  [('abday', ('teo', 'cei', 'ome', 'yei', 'nau', 'mac', 'chi')),
   ('day', ('teoilhuitl', 'ceilhuitl', 'omeilhuitl', 'yeilhuitl', 'nahuilhuitl', 'macuililhuitl', 'chicuaceilhuitl')),
   ('abmon', ('ene', 'feb', 'mar', 'abr', 'may', 'jun', 'jul', 'ago', 'sep', 'oct', 'nov', 'dic')),
   ('mon', ('enero', 'febrero', 'marzo', 'abril', 'mayo', 'junio', 'julio', 'agosto', 'septiembre', 'octubre', 'noviembre', 'diciembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', '\u2009'),
   ('yesexpr', '^[sSqQyY].*'),
   ('noexpr', '^[nNaA].*')]),
 ('niu_NU',
  [('abday', ('Tapu', 'Gofua', 'Ua', 'Lotu', 'Tuloto', 'Falaile', 'Faiumu')),
   ('day', ('Aho Tapu', 'Aho Gofua', 'Aho Ua', 'Aho Lotu', 'Aho Tuloto', 'Aho Falaile', 'Aho Faiumu')),
   ('abmon', ('Ian', 'Fep', 'Mas', 'Ape', 'Me', 'Iun', 'Iul', 'Aok', 'Sep', 'Oke', 'Nov', 'Tes')),
   ('mon', ('Ianuali', 'Fepuali', 'Masi', 'Apelila', 'Me', 'Iuni', 'Iulai', 'Aokuso', 'Sepetema', 'Oketopa', 'Novema', 'Tesemo')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ēĒyY].*'),
   ('noexpr', '^[nN].*')]),
 ('niu_NZ',
  [('abday', ('Tapu', 'Gofua', 'Ua', 'Lotu', 'Tuloto', 'Falaile', 'Faiumu')),
   ('day', ('Aho Tapu', 'Aho Gofua', 'Aho Ua', 'Aho Lotu', 'Aho Tuloto', 'Aho Falaile', 'Aho Faiumu')),
   ('abmon', ('Ian', 'Fep', 'Mas', 'Ape', 'Me', 'Iun', 'Iul', 'Aok', 'Sep', 'Oke', 'Nov', 'Tes')),
   ('mon', ('Ianuali', 'Fepuali', 'Masi', 'Apelila', 'Me', 'Iuni', 'Iulai', 'Aokuso', 'Sepetema', 'Oketopa', 'Novema', 'Tesemo')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ēĒyY].*'),
   ('noexpr', '^[nN].*')]),
 ('nl_AW',
  [('abday', ('zo', 'ma', 'di', 'wo', 'do', 'vr', 'za')),
   ('day', ('zondag', 'maandag', 'dinsdag', 'woensdag', 'donderdag', 'vrijdag', 'zaterdag')),
   ('abmon', ('jan', 'feb', 'mrt', 'apr', 'mei', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januari', 'februari', 'maart', 'april', 'mei', 'juni', 'juli', 'augustus', 'september', 'oktober', 'november', 'december')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('nl_BE',
  [('abday', ('zo', 'ma', 'di', 'wo', 'do', 'vr', 'za')),
   ('day', ('zondag', 'maandag', 'dinsdag', 'woensdag', 'donderdag', 'vrijdag', 'zaterdag')),
   ('abmon', ('jan', 'feb', 'mrt', 'apr', 'mei', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januari', 'februari', 'maart', 'april', 'mei', 'juni', 'juli', 'augustus', 'september', 'oktober', 'november', 'december')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('nl_NL',
  [('abday', ('zo', 'ma', 'di', 'wo', 'do', 'vr', 'za')),
   ('day', ('zondag', 'maandag', 'dinsdag', 'woensdag', 'donderdag', 'vrijdag', 'zaterdag')),
   ('abmon', ('jan', 'feb', 'mrt', 'apr', 'mei', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januari', 'februari', 'maart', 'april', 'mei', 'juni', 'juli', 'augustus', 'september', 'oktober', 'november', 'december')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('nn_NO',
  [('abday', ('su.', 'må.', 'ty.', 'on.', 'to.', 'fr.', 'la.')),
   ('day', ('sundag ', 'måndag ', 'tysdag ', 'onsdag ', 'torsdag ', 'fredag ', 'laurdag ')),
   ('abmon', ('jan.', 'feb.', 'mars', 'april', 'mai', 'juni', 'juli', 'aug.', 'sep.', 'okt.', 'nov.', 'des.')),
   ('mon', ('januar', 'februar', 'mars', 'april', 'mai', 'juni', 'juli', 'august', 'september', 'oktober', 'november', 'desember')),
   ('d_t_fmt', '%a %d. %b %Y kl. %H.%M %z'),
   ('d_fmt', '%d. %b %Y'),
   ('t_fmt', 'kl. %H.%M %z'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[JjYy].*'),
   ('noexpr', '^[Nn].*')]),
 ('nr_ZA',
  [('abday', ('Son', 'Mvu', 'Bil', 'Tha', 'Ne', 'Hla', 'Gqi')),
   ('day', ('uSonto', 'uMvulo', 'uLesibili', 'lesithathu', 'uLesine', 'ngoLesihlanu', 'umGqibelo')),
   ('abmon', ('Jan', 'Feb', 'Mat', 'Apr', 'Mey', 'Jun', 'Jul', 'Arh', 'Sep', 'Okt', 'Usi', 'Dis')),
   ('mon', ('Janabari', 'uFeberbari', 'uMatjhi', 'u-Apreli', 'Meyi', 'Juni', 'Julayi', 'Arhostosi', 'Septemba', 'Oktoba', 'Usinyikhaba', 'Disemba')),
   ('d_t_fmt', '%a %-e %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('nso_ZA',
  [('abday', ('Son', 'Moš', 'Bed', 'Rar', 'Ne', 'Hla', 'Mok')),
   ('day', ('LaMorena', 'Mošupologo', 'Labobedi', 'Laboraro', 'Labone', 'Labohlano', 'Mokibelo')),
   ('abmon', ('Jan', 'Feb', 'Mat', 'Apr', 'Mei', 'Jun', 'Jul', 'Ago', 'Set', 'Okt', 'Nof', 'Dis')),
   ('mon', ('Janaware', 'Febereware', 'Matšhe', 'Aprele', 'Mei', 'June', 'Julae', 'Agostose', 'Setemere', 'Oktobere', 'Nofemere', 'Disemere')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYeE]'),
   ('noexpr', '^[nNaA]')]),
 ('oc_FR',
  [('abday', ('dim', 'lun', 'mar', 'mec', 'jòu', 'ven', 'sab')),
   ('day', ('dimenge', 'diluns', 'dimars', 'dimecres', 'dijóus', 'divendres', 'disabte')),
   ('abmon', ('gen', 'feb', 'mar', 'abr', 'mai', 'jun', 'jul', 'ago', 'set', 'oct', 'nov', 'dec')),
   ('mon', ('genièr', 'febrièr', 'març', 'abrial', 'mai', 'junh', 'julhet', 'agost', 'setembre', 'octobre', 'novembre', 'decembre')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[oOsSyY].*'),
   ('noexpr', '^[nN].*')]),
 ('om_ET',
  [('abday', ('Dil', 'Wix', 'Qib', 'Rob', 'Kam', 'Jim', 'San')),
   ('day', ('Dilbata', 'Wiixata', 'Qibxata', 'Roobii', 'Kamiisa', 'Jimaata', 'Sanbata')),
   ('abmon', ('Ama', 'Gur', 'Bit', 'Elb', 'Cam', 'Wax', 'Ado', 'Hag', 'Ful', 'Onk', 'Sad', 'Mud')),
   ('mon',
    ('Amajjii', 'Guraandhala', 'Bitooteessa', 'Elba', 'Caamsa', 'Waxabajjii', 'Adooleessa', 'Hagayya', 'Fuulbana', 'Onkololeessa', 'Sadaasa', 'Muddee')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('om_KE',
  [('abday', ('Dil', 'Wix', 'Qib', 'Rob', 'Kam', 'Jim', 'San')),
   ('day', ('Dilbata', 'Wiixata', 'Qibxata', 'Roobii', 'Kamiisa', 'Jimaata', 'Sanbata')),
   ('abmon', ('Ama', 'Gur', 'Bit', 'Elb', 'Cam', 'Wax', 'Ado', 'Hag', 'Ful', 'Onk', 'Sad', 'Mud')),
   ('mon',
    ('Amajjii', 'Guraandhala', 'Bitooteessa', 'Elba', 'Caamsa', 'Waxabajjii', 'Adooleessa', 'Hagayya', 'Fuulbana', 'Onkololeessa', 'Sadaasa', 'Muddee')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('or_IN',
  [('abday', ('ରବି', 'ସୋମ', 'ମଙ୍ଗଳ', 'ବୁଧ', 'ଗୁରୁ', 'ଶୁକ୍ର', 'ଶନି')),
   ('day', ('ରବିବାର', 'ସୋମବାର', 'ମଙ୍ଗଳବାର', 'ବୁଧବାର', 'ଗୁରୁବାର', 'ଶୁକ୍ରବାର', 'ଶନିବାର')),
   ('abmon', ('1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11', '12')),
   ('mon', ('ଜାନୁଆରୀ', 'ଫେବୃଆରୀ', 'ମାର୍ଚ୍ଚ', 'ଅପ୍ରେଲ', 'ମଇ', 'ଜୁନ', 'ଜୁଲାଇ', 'ଅଗଷ୍ଟ', 'ସେପ୍ଟେମ୍ବର', 'ଅକ୍ଟୋବର', 'ନଭେମ୍ବର', 'ଡିସେମ୍ବର')),
   ('d_t_fmt', '%Oe %B %Oy %OI:%OM:%OS %p %Z'),
   ('d_fmt', '%Od-%Om-%Oy'),
   ('t_fmt', '%OI:%OM:%OS %p'),
   ('t_fmt_ampm', '%OI:%OM:%OS %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('os_RU',
  [('abday', ('Хцб', 'Крс', 'Дцг', 'Æрт', 'Цпр', 'Мрб', 'Сбт')),
   ('day', ('Хуыцаубон', 'Къуырисæр', 'Дыццæг', 'Æртыццæг', 'Цыппæрæм', 'Майрæмбон', 'Сабат')),
   ('abmon', ('Янв', 'Фев', 'Мар', 'Апр', 'Май', 'Июн', 'Июл', 'Авг', 'Сен', 'Окт', 'Ноя', 'Дек')),
   ('mon', ('Январь', 'Февраль', 'Мартъи', 'Апрель', 'Май', 'Июнь', 'Июль', 'Август', 'Сентябрь', 'Октябрь', 'Ноябрь', 'Декабрь')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[УдYy].*'),
   ('noexpr', '^[унNn].*')]),
 ('pa_IN',
  [('abday', ('ਐਤ ', 'ਸੋਮ ', 'ਮੰਗਲ ', 'ਬੁੱਧ ', 'ਵੀਰ ', 'ਸ਼ੁੱਕਰ ', 'ਸ਼ਨਿੱਚਰ ')),
   ('day', ('ਐਤਵਾਰ ', 'ਸੋਮਵਾਰ ', 'ਮੰਗਲਵਾਰ ', 'ਬੁੱਧਵਾਰ ', 'ਵੀਰਵਾਰ ', 'ਸ਼ੁੱਕਰਵਾਰ ', 'ਸ਼ਨਿੱਚਰਵਾਰ ')),
   ('abmon', ('ਜਨਵਰੀ', 'ਫ਼ਰਵਰੀ', 'ਮਾਰਚ', 'ਅਪਰੈਲ', 'ਮਈ', 'ਜੂਨ', 'ਜੁਲਾਈ', 'ਅਗਸਤ', 'ਸਤੰਬਰ', 'ਅਕਤੂਬਰ', 'ਨਵੰਬਰ', 'ਦਸੰਬਰ')),
   ('mon', ('ਜਨਵਰੀ', 'ਫ਼ਰਵਰੀ', 'ਮਾਰਚ', 'ਅਪਰੈਲ', 'ਮਈ', 'ਜੂਨ', 'ਜੁਲਾਈ', 'ਅਗਸਤ', 'ਸਤੰਬਰ', 'ਅਕਤੂਬਰ', 'ਨਵੰਬਰ', 'ਦਸੰਬਰ')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('pa_PK',
  [('abday', ('اتوار', 'پير', 'منگل', 'بدھ', 'جمعرات', 'جمعه', 'هفته')),
   ('day', ('اتوار', 'پير', 'منگل', 'بدھ', 'جمعرات', 'جمعه', 'هفته')),
   ('abmon', ('جنوري', 'فروري', 'مارچ', 'اپريل', 'مٓی', 'جون', 'جولاي', 'اگست', 'ستمبر', 'اكتوبر', 'نومبر', 'دسمبر')),
   ('mon', ('جنوري', 'فروري', 'مارچ', 'اپريل', 'مٓی', 'جون', 'جولاي', 'اگست', 'ستمبر', 'اكتوبر', 'نومبر', 'دسمبر')),
   ('d_t_fmt', 'و %H:%M:%S %Z ت %d %B %Y'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%P %I:%M:%S'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYهبf].*'),
   ('noexpr', '^[nNنo].*')]),
 ('pap_AN',
  [('abday', ('do', 'lu', 'ma', 'we', 'ra', 'bi', 'sa')),
   ('day', ('Djadomingo', 'Djaluna', 'Djamars', 'Djawebs', 'Djarason', 'Djabierne', 'Djasabra')),
   ('abmon', ('Yan', 'Feb', 'Mar', 'Apr', 'Mei', 'Yün', 'Yül', 'Oug', 'Sèp', 'Okt', 'Nov', 'Des')),
   ('mon', ('Yanüari', 'Febrüari', 'Mart', 'Aprel', 'Mei', 'Yüni', 'Yüli', 'Ougùstùs', 'Sèptèmber', 'Oktober', 'Novèmber', 'Desèmber')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('pap_AW',
  [('abday', ('do', 'lu', 'ma', 'we', 'ra', 'bi', 'sa')),
   ('day', ('Djadomingo', 'Djaluna', 'Djamars', 'Djawebs', 'Djarason', 'Djabierne', 'Djasabra')),
   ('abmon', ('Yan', 'Feb', 'Mar', 'Apr', 'Mei', 'Yün', 'Yül', 'Oug', 'Sèp', 'Okt', 'Nov', 'Des')),
   ('mon', ('Yanüari', 'Febrüari', 'Mart', 'Aprel', 'Mei', 'Yüni', 'Yüli', 'Ougùstùs', 'Sèptèmber', 'Oktober', 'Novèmber', 'Desèmber')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('pap_CW',
  [('abday', ('do', 'lu', 'ma', 'we', 'ra', 'bi', 'sa')),
   ('day', ('Djadomingo', 'Djaluna', 'Djamars', 'Djawebs', 'Djarason', 'Djabierne', 'Djasabra')),
   ('abmon', ('Yan', 'Feb', 'Mar', 'Apr', 'Mei', 'Yün', 'Yül', 'Oug', 'Sèp', 'Okt', 'Nov', 'Des')),
   ('mon', ('Yanüari', 'Febrüari', 'Mart', 'Aprel', 'Mei', 'Yüni', 'Yüli', 'Ougùstùs', 'Sèptèmber', 'Oktober', 'Novèmber', 'Desèmber')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('pl_PL',
  [('abday', ('nie', 'pon', 'wto', 'śro', 'czw', 'pią', 'sob')),
   ('day', ('niedziela', 'poniedziałek', 'wtorek', 'środa', 'czwartek', 'piątek', 'sobota')),
   ('abmon', ('sty', 'lut', 'mar', 'kwi', 'maj', 'cze', 'lip', 'sie', 'wrz', 'paź', 'lis', 'gru')),
   ('mon', ('styczeń', 'luty', 'marzec', 'kwiecień', 'maj', 'czerwiec', 'lipiec', 'sierpień', 'wrzesień', 'październik', 'listopad', 'grudzień')),
   ('d_t_fmt', '%a, %-d %b %Y, %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[TtYy].*'),
   ('noexpr', '^[nN].*')]),
 ('ps_AF',
  [('abday', ('ی.', 'د.', 'س.', 'چ.', 'پ.', 'ج.', 'ش.')),
   ('day', ('یکشنبه', 'دوشنبه', 'سه\u200cشنبه', 'چارشنبه', 'پنجشنبه', 'جمعه', 'شنبه')),
   ('abmon', ('جنو', 'فبر', 'مار', 'اپر', 'مـې', 'جون', 'جول', 'اګس', 'سپت', 'اکت', 'نوم', 'دسم')),
   ('mon', ('جنوري', 'فبروري', 'مارچ', 'اپریل', 'مې', 'جون', 'جولاي', 'اګست', 'سپتمبر', 'اکتوبر', 'نومبر', 'دسمبر')),
   ('d_t_fmt', '%A د %Y د %B %e، %Z %H:%M:%S'),
   ('d_fmt', 'د %Y د %B %e'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '\u202b%I:%M:%S %p\u202c'),
   ('radixchar', '٫'),
   ('thousep', '٬'),
   ('yesexpr', '^[yYبf].*'),
   ('noexpr', '^[nNخنo].*')]),
 ('pt_BR',
  [('abday', ('Dom', 'Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sáb')),
   ('day', ('domingo', 'segunda', 'terça', 'quarta', 'quinta', 'sexta', 'sábado')),
   ('abmon', ('Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set', 'Out', 'Nov', 'Dez')),
   ('mon', ('janeiro', 'fevereiro', 'março', 'abril', 'maio', 'junho', 'julho', 'agosto', 'setembro', 'outubro', 'novembro', 'dezembro')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[SsyY].*'),
   ('noexpr', '^[nN].*')]),
 ('pt_PT',
  [('abday', ('Dom', 'Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sáb')),
   ('day', ('Domingo', 'Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta', 'Sábado')),
   ('abmon', ('Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set', 'Out', 'Nov', 'Dez')),
   ('mon', ('Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[SsyY].*'),
   ('noexpr', '^[nN].*')]),
 ('quz_PE',
  [('abday', ('tum', 'lun', 'mar', 'miy', 'juy', 'wiy', 'saw')),
   ('day', ('tuminku', 'lunis', 'martis', 'miyirkulis', 'juywis', 'wiyirnis', 'sawatu')),
   ('abmon', ('ini', 'phi', 'mar', 'awr', 'may', 'hun', 'hul', 'agu', 'sip', 'ukt', 'nuw', 'tis')),
   ('mon', ('iniru', 'phiwriru', 'marsu', 'awril', 'mayu', 'huniyu', 'huliyu', 'agustu', 'siptiyimri', 'uktuwri', 'nuwiyimri', 'tisiyimri')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%I:%M:%S %p'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[aAsSyY].*'),
   ('noexpr', '^[mMnN].*')]),
 ('ro_RO',
  [('abday', ('Du', 'Lu', 'Ma', 'Mi', 'Jo', 'Vi', 'Sb')),
   ('day', ('duminică', 'luni', 'marţi', 'miercuri', 'joi', 'vineri', 'sâmbătă')),
   ('abmon', ('ian', 'feb', 'mar', 'apr', 'mai', 'iun', 'iul', 'aug', 'sep', 'oct', 'nov', 'dec')),
   ('mon', ('ianuarie', 'februarie', 'martie', 'aprilie', 'mai', 'iunie', 'iulie', 'august', 'septembrie', 'octombrie', 'noiembrie', 'decembrie')),
   ('d_t_fmt', '%a %d %b %Y %T %z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[DdYy].*'),
   ('noexpr', '^[nN].*')]),
 ('ru_RU',
  [('abday', ('Вс', 'Пн', 'Вт', 'Ср', 'Чт', 'Пт', 'Сб')),
   ('day', ('Воскресенье', 'Понедельник', 'Вторник', 'Среда', 'Четверг', 'Пятница', 'Суббота')),
   ('abmon', ('янв', 'фев', 'мар', 'апр', 'май', 'июн', 'июл', 'авг', 'сен', 'окт', 'ноя', 'дек')),
   ('mon', ('Январь', 'Февраль', 'Март', 'Апрель', 'Май', 'Июнь', 'Июль', 'Август', 'Сентябрь', 'Октябрь', 'Ноябрь', 'Декабрь')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[ДдYy].*'),
   ('noexpr', '^[НнNn].*')]),
 ('ru_UA',
  [('abday', ('Вск', 'Пнд', 'Вто', 'Срд', 'Чтв', 'Птн', 'Суб')),
   ('day', ('Воскресенье', 'Понедельник', 'Вторник', 'Среда', 'Четверг', 'Пятница', 'Суббота')),
   ('abmon', ('Янв', 'Фев', 'Мар', 'Апр', 'Май', 'Июн', 'Июл', 'Авг', 'Сен', 'Окт', 'Ноя', 'Дек')),
   ('mon', ('Январь', 'Февраль', 'Март', 'Апрель', 'Май', 'Июнь', 'Июль', 'Август', 'Сентябрь', 'Октябрь', 'Ноябрь', 'Декабрь')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[ДдYy].*'),
   ('noexpr', '^[НнNn].*')]),
 ('rw_RW',
  [('abday', ('Mwe', 'Mbe', 'Kab', 'Gtu', 'Kan', 'Gnu', 'Gnd')),
   ('day', ('Ku cyumweru', 'Kuwa mbere', 'Kuwa kabiri', 'Kuwa gatatu', 'Kuwa kane', 'Kuwa gatanu', 'Kuwa gatandatu')),
   ('abmon', ('Mut', 'Gas', 'Wer', 'Mat', 'Gic', 'Kam', 'Nya', 'Kan', 'Nze', 'Ukw', 'Ugu', 'Uku')),
   ('mon', ('Mutarama', 'Gashyantare', 'Werurwe', 'Mata', 'Gicuransi', 'Kamena', 'Nyakanga', 'Kanama', 'Nzeli', 'Ukwakira', 'Ugushyingo', 'Ukuboza')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nNoO]')]),
 ('sa_IN',
  [('abday', ('रविः', 'सोम:', 'मंगल:', 'बुध:', 'बृहस्पतिः', 'शुक्र', 'शनि:')),
   ('day', ('रविवासर:', 'सोमवासर:', 'मंगलवासर:', 'बुधवासर:', 'बृहस्पतिवासरः', 'शुक्रवासर', 'शनिवासर:')),
   ('abmon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[aAyY].*'),
   ('noexpr', '^[nN].*')]),
 ('sat_IN',
  [('abday', ('सिंगे', 'ओते', 'बाले', 'सागुन', 'सारदी', 'जारुम', 'ञुहुम')),
   ('day', ('सिंगेमाँहाँ', 'ओतेमाँहाँ', 'बालेमाँहाँ', 'सागुनमाँहाँ', 'सारदीमाँहाँ', 'जारुममाँहाँ', 'ञुहुममाँहाँ')),
   ('abmon', ('जनवरी', 'फरवरी', 'मार्च', 'अप्रेल', 'मई', 'जुन', 'जुलाई', 'अगस्त', 'सितम्बर', 'अखथबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फरवरी', 'मार्च', 'अप्रेल', 'मई', 'जुन', 'जुलाई', 'अगस्त', 'सितम्बर', 'अखथबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^(होय|[yY])'),
   ('noexpr', '^(बाङ|[nN])')]),
 ('sc_IT',
  [('abday', ('Dom', 'Lun', 'Mar', 'Mèr', 'Jòb', 'Cen', 'Sàb')),
   ('day', ('Domìngu', 'Lunis', 'Martis', 'Mèrcuris', 'Jòbia', 'Cenàbara', 'Sàbadu')),
   ('abmon', ('Gen', 'Fri', 'Mar', 'Abr', 'May', 'Làm', 'Arj', 'Aus', 'Cab', 'Lad', 'Don', 'Ida')),
   ('mon', ('Gennarju', 'Friarju', 'Martzu', 'Abrili', 'Mayu', 'Làmpadas', 'Arjolas', 'Austu', 'Cabudanni', 'Ladàmini', 'Donnyasantu', 'Idas')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d. %m. %y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[sSjJoOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('sd_IN',
  [('abday', ('آرتوارُ', 'سومرُ', 'منگلُ', 'ٻُڌرُ', 'وسپت', 'جُمو', 'ڇنڇر')),
   ('day', ('آرتوارُ', 'سومرُ', 'منگلُ', 'ٻُڌرُ', 'وسپت', 'جُمو', 'ڇنڇر')),
   ('abmon', ('جنوري', 'فبروري', 'مارچ', 'اپريل', 'مي', 'جون', 'جولاءِ', 'آگسٽ', 'سيپٽيمبر', 'آڪٽوبر', 'نومبر', 'ڊسمبر')),
   ('mon', ('جنوري', 'فبروري', 'مارچ', 'اپريل', 'مي', 'جون', 'جولاءِ', 'آگسٽ', 'سيپٽيمبر', 'آڪٽوبر', 'نومبر', 'ڊسمبر')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[نyY].*'),
   ('noexpr', '^[لnN].*')]),
 ('se_NO',
  [('abday', ('sotn', 'vuos', 'maŋ', 'gask', 'duor', 'bear', 'láv')),
   ('day', ('sotnabeaivi', 'vuossárga', 'maŋŋebarga', 'gaskavahkku', 'duorasdat', 'bearjadat', 'lávvardat')),
   ('abmon', ('ođđj', 'guov', 'njuk', 'cuoŋ', 'mies', 'geas', 'suoi', 'borg', 'čakč', 'golg', 'skáb', 'juov')),
   ('mon',
    ('ođđajagemánu',
     'guovvamánu',
     'njukčamánu',
     'cuoŋománu',
     'miessemánu',
     'geassemánu',
     'suoidnemánu',
     'borgemánu',
     'čakčamánu',
     'golggotmánu',
     'skábmamánu',
     'juovlamánu')),
   ('d_t_fmt', '%a, %b %e. b. %Y %T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[JjYy].*'),
   ('noexpr', '^[Ii].*')]),
 ('shs_CA',
  [('abday', ('Sxe', 'Spe', 'Sel', 'Ske', 'Sme', 'Sts', 'Stq')),
   ('day', ('Sxetspesq̓t', 'Spetkesq̓t', 'Selesq̓t', 'Skellesq̓t', 'Smesesq̓t', 'Stselkstesq̓t', 'Stqmekstesq̓t')),
   ('abmon', ('Kwe', 'Tsi', 'Sqe', 'Éwt', 'Ell', 'Tsp', 'Tqw', 'Ct̓é', 'Qel', 'Wél', 'U7l', 'Tet')),
   ('mon',
    ('Pellkwet̓min',
     'Pelctsipwen̓ten',
     'Pellsqépts',
     'Peslléwten',
     'Pell7ell7é7llqten',
     'Pelltspéntsk',
     'Pelltqwelq̓wél̓t',
     'Pellct̓éxel̓cten',
     'Pesqelqlélten',
     'Pesllwélsten',
     'Pellc7ell7é7llcwten̓',
     'Pelltetétq̓em')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYoO].*'),
   ('noexpr', '^[nN].*')]),
 ('si_LK',
  [('abday', ('ඉ', 'ස', 'අ', 'බ', 'බ්\u200dර', 'සි', 'සෙ')),
   ('day', ('ඉරිදා', 'සඳුදා', 'අඟහරුවාදා', 'බදාදා', 'බ්\u200dරහස්පතින්දා', 'සිකුරාදා', 'සෙනසුරාදා')),
   ('abmon', ('ජන', 'පෙබ', 'මාර්', 'අප්\u200dරි', 'මැයි', 'ජූනි', 'ජූලි', 'අගෝ', 'සැප්', 'ඔක්', 'නෙවැ', 'දෙසැ')),
   ('mon', ('ජනවාරි', 'පෙබරවාරි', 'මාර්තු', 'අප්\u200dරියෙල්', 'මැයි', 'ජූනි', 'ජූලි', 'අගෝස්තු', 'සැප්තැම්බර්', 'ඔක්තෝබර්', 'නොවැම්බර්', 'දෙසැම්බර්')),
   ('d_t_fmt', '%Y-%m-%d %H:%M:%S %z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%p %I:%M:%S'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ඔYy]'),
   ('noexpr', '^[නNn]')]),
 ('sid_ET',
  [('abday', ('Sam', 'San', 'Mak', 'Row', 'Ham', 'Arb', 'Qid')),
   ('day', ('Sambata', 'Sanyo', 'Maakisanyo', 'Roowe', 'Hamuse', 'Arbe', 'Qidaame')),
   ('abmon', ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec')),
   ('mon', ('January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('sk_SK',
  [('abday', ('Ne', 'Po', 'Ut', 'St', 'Št', 'Pi', 'So')),
   ('day', ('Nedeľa', 'Pondelok', 'Utorok', 'Streda', 'Štvrtok', 'Piatok', 'Sobota')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'máj', 'jún', 'júl', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('január', 'február', 'marec', 'apríl', 'máj', 'jún', 'júl', 'august', 'september', 'október', 'november', 'december')),
   ('d_t_fmt', '%a\xa0%e.\xa0%B\xa0%Y,\xa0%H:%M:%S\xa0%Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S'),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[aAáÁyY].*'),
   ('noexpr', '^[nN].*')]),
 ('sl_SI',
  [('abday', ('ned', 'pon', 'tor', 'sre', 'čet', 'pet', 'sob')),
   ('day', ('nedelja', 'ponedeljek', 'torek', 'sreda', 'četrtek', 'petek', 'sobota')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'maj', 'jun', 'jul', 'avg', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januar', 'februar', 'marec', 'april', 'maj', 'junij', 'julij', 'avgust', 'september', 'oktober', 'november', 'december')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d. %m. %Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[YyJj].*'),
   ('noexpr', '^[Nn].*')]),
 ('so_DJ',
  [('abday', ('axa', 'isn', 'sal', 'arb', 'kha', 'jim', 'sab')),
   ('day', ('Axad', 'Isniin', 'Salaaso', 'Arbaco', 'Khamiis', 'Jimco', 'Sabti')),
   ('abmon', ('kob', 'lab', 'sad', 'afr', 'sha', 'lix', 'tod', 'sid', 'sag', 'tob', 'kit', 'lit')),
   ('mon',
    ('Bisha Koobaad',
     'Bisha Labaad',
     'Bisha Saddexaad',
     'Bisha Afraad',
     'Bisha Shanaad',
     'Bisha Lixaad',
     'Bisha Todobaad',
     'Bisha Sideedaad',
     'Bisha Sagaalaad',
     'Bisha Tobnaad',
     'Bisha Kow iyo Tobnaad',
     'Bisha Laba iyo Tobnaad')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[oOyY].*'),
   ('noexpr', '^[nN].*')]),
 ('so_ET',
  [('abday', ('Axa', 'Isn', 'Sal', 'Arb', 'Kha', 'Jim', 'Sab')),
   ('day', ('Axad', 'Isniin', 'Salaaso', 'Arbaco', 'Khamiis', 'Jimco', 'Sabti')),
   ('abmon', ('Kob', 'Lab', 'Sad', 'Afr', 'Sha', 'Lix', 'Tod', 'Sid', 'Sag', 'Tob', 'KIT', 'LIT')),
   ('mon',
    ('Bisha Koobaad',
     'Bisha Labaad',
     'Bisha Saddexaad',
     'Bisha Afraad',
     'Bisha Shanaad',
     'Bisha Lixaad',
     'Bisha Todobaad',
     'Bisha Sideedaad',
     'Bisha Sagaalaad',
     'Bisha Tobnaad',
     'Bisha Kow iyo Tobnaad',
     'Bisha Laba iyo Tobnaad')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('so_KE',
  [('abday', ('Axa', 'Isn', 'Sal', 'Arb', 'Kha', 'Jim', 'Sab')),
   ('day', ('Axad', 'Isniin', 'Salaaso', 'Arbaco', 'Khamiis', 'Jimco', 'Sabti')),
   ('abmon', ('Kob', 'Lab', 'Sad', 'Afr', 'Sha', 'Lix', 'Tod', 'Sid', 'Sag', 'Tob', 'KIT', 'LIT')),
   ('mon',
    ('Bisha Koobaad',
     'Bisha Labaad',
     'Bisha Saddexaad',
     'Bisha Afraad',
     'Bisha Shanaad',
     'Bisha Lixaad',
     'Bisha Todobaad',
     'Bisha Sideedaad',
     'Bisha Sagaalaad',
     'Bisha Tobnaad',
     'Bisha Kow iyo Tobnaad',
     'Bisha Laba iyo Tobnaad')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('so_SO',
  [('abday', ('Axa', 'Isn', 'Sal', 'Arb', 'Kha', 'Jim', 'Sab')),
   ('day', ('Axad', 'Isniin', 'Salaaso', 'Arbaco', 'Khamiis', 'Jimco', 'Sabti')),
   ('abmon', ('Kob', 'Lab', 'Sad', 'Afr', 'Sha', 'Lix', 'Tod', 'Sid', 'Sag', 'Tob', 'KIT', 'LIT')),
   ('mon',
    ('Bisha Koobaad',
     'Bisha Labaad',
     'Bisha Saddexaad',
     'Bisha Afraad',
     'Bisha Shanaad',
     'Bisha Lixaad',
     'Bisha Todobaad',
     'Bisha Sideedaad',
     'Bisha Sagaalaad',
     'Bisha Tobnaad',
     'Bisha Kow iyo Tobnaad',
     'Bisha Laba iyo Tobnaad')),
   ('d_t_fmt', '%A, %B %e, %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('sq_AL',
  [('abday', ('Die ', 'Hën ', 'Mar ', 'Mër ', 'Enj ', 'Pre ', 'Sht ')),
   ('day', ('e diel ', 'e hënë ', 'e martë ', 'e mërkurë ', 'e enjte ', 'e premte ', 'e shtunë ')),
   ('abmon', ('Jan', 'Shk', 'Mar', 'Pri', 'Maj', 'Qer', 'Kor', 'Gsh', 'Sht', 'Tet', 'Nën', 'Dhj')),
   ('mon', ('janar', 'shkurt', 'mars', 'prill', 'maj', 'qershor', 'korrik', 'gusht', 'shtator', 'tetor', 'nëntor', 'dhjetor')),
   ('d_t_fmt', '%Y-%b-%d %I.%M.%S.%p %Z'),
   ('d_fmt', '%Y-%b-%d'),
   ('t_fmt', '%I.%M.%S. %Z'),
   ('t_fmt_ampm', '%I.%M.%S.%p %Z'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[yYpP].*'),
   ('noexpr', '^[nNjJ].*')]),
 ('sq_MK',
  [('abday', ('Die ', 'Hën ', 'Mar ', 'Mër ', 'Enj ', 'Pre ', 'Sht ')),
   ('day', ('e diel ', 'e hënë ', 'e martë ', 'e mërkurë ', 'e enjte ', 'e premte ', 'e shtunë ')),
   ('abmon', ('Jan', 'Shk', 'Mar', 'Pri', 'Maj', 'Qer', 'Kor', 'Gsh', 'Sht', 'Tet', 'Nën', 'Dhj')),
   ('mon', ('janar', 'shkurt', 'mars', 'prill', 'maj', 'qershor', 'korrik', 'gusht', 'shtator', 'tetor', 'nëntor', 'dhjetor')),
   ('d_t_fmt', '%Y-%b-%d %I.%M.%S.%p %Z'),
   ('d_fmt', '%Y-%b-%d'),
   ('t_fmt', '%I.%M.%S. %Z'),
   ('t_fmt_ampm', '%I.%M.%S.%p %Z'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[yYpP].*'),
   ('noexpr', '^[nNjJ].*')]),
 ('sr_ME',
  [('abday', ('нед', 'пон', 'уто', 'сри', 'чет', 'пет', 'суб')),
   ('day', ('недјеља', 'понедељак', 'уторак', 'сриједа', 'четвртак', 'петак', 'субота')),
   ('abmon', ('јан', 'феб', 'мар', 'апр', 'мај', 'јун', 'јул', 'авг', 'сеп', 'окт', 'нов', 'дец')),
   ('mon', ('јануар', 'фебруар', 'март', 'април', 'мај', 'јун', 'јул', 'август', 'септембар', 'октобар', 'новембар', 'децембар')),
   ('d_t_fmt', '%A, %d. %B %Y. %T %Z'),
   ('d_fmt', '%d.%m.%Y.'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%T'),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[ДдDdYy]'),
   ('noexpr', '^[НнNn]')]),
 ('sr_RS',
  [('abday', ('нед', 'пон', 'уто', 'сре', 'чет', 'пет', 'суб')),
   ('day', ('недеља', 'понедељак', 'уторак', 'среда', 'четвртак', 'петак', 'субота')),
   ('abmon', ('јан', 'феб', 'мар', 'апр', 'мај', 'јун', 'јул', 'авг', 'сеп', 'окт', 'нов', 'дец')),
   ('mon', ('јануар', 'фебруар', 'март', 'април', 'мај', 'јун', 'јул', 'август', 'септембар', 'октобар', 'новембар', 'децембар')),
   ('d_t_fmt', '%A, %d. %B %Y. %T %Z'),
   ('d_fmt', '%d.%m.%Y.'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%T'),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[ДдDdYy]'),
   ('noexpr', '^[НнNn]')]),
 ('ss_ZA',
  [('abday', ('Son', 'Mso', 'Bil', 'Tsa', 'Ne', 'Hla', 'Mgc')),
   ('day', ('Lisontfo', 'uMsombuluko', 'Lesibili', 'Lesitsatfu', 'Lesine', 'Lesihlanu', 'uMgcibelo')),
   ('abmon', ('Bhi', 'Van', 'Vul', 'Mab', 'Khk', 'Nhl', 'Kho', 'Ngc', 'Nyo', 'Imp', 'Lwe', 'Ngo')),
   ('mon',
    ('Bhimbidvwane', 'iNdlovane', 'iNdlovulenkhulu', 'Mabasa', 'Inkhwenkhweti', 'iNhlaba', 'Kholwane', 'iNgci', 'iNyoni', 'Impala', 'Lweti', 'iNgongoni')),
   ('d_t_fmt', '%a %-e %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nNaA]')]),
 ('st_ZA',
  [('abday', ('Son', 'Mma', 'Bed', 'Rar', 'Ne', 'Hla', 'Moq')),
   ('day', ('Sontaha', 'Mantaha', 'Labobedi', 'Laboraro', 'Labone', 'Labohlano', 'Moqebelo')),
   ('abmon', ('Phe', 'Hla', 'TlH', 'Mme', 'Mot', 'Jan', 'Upu', 'Pha', 'Leo', 'Mph', 'Pud', 'Tsh')),
   ('mon', ('Pherekgong', 'Hlakola', 'Tlhakubele', 'Mmese', 'Motsheanong', 'Phupjane', 'Phupu', 'Phato', 'Leotse', 'Mphalane', 'Pudungwana', 'Tshitwe')),
   ('d_t_fmt', '%a %-e %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('sv_FI',
  [('abday', ('sön', 'mån', 'tis', 'ons', 'tor', 'fre', 'lör')),
   ('day', ('söndag', 'måndag', 'tisdag', 'onsdag', 'torsdag', 'fredag', 'lördag')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'maj', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januari', 'februari', 'mars', 'april', 'maj', 'juni', 'juli', 'augusti', 'september', 'oktober', 'november', 'december')),
   ('d_t_fmt', '%a %e. %B %Y %H.%M.%S'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%H.%M.%S'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('sv_SE',
  [('abday', ('sön', 'mån', 'tis', 'ons', 'tor', 'fre', 'lör')),
   ('day', ('söndag', 'måndag', 'tisdag', 'onsdag', 'torsdag', 'fredag', 'lördag')),
   ('abmon', ('jan', 'feb', 'mar', 'apr', 'maj', 'jun', 'jul', 'aug', 'sep', 'okt', 'nov', 'dec')),
   ('mon', ('januari', 'februari', 'mars', 'april', 'maj', 'juni', 'juli', 'augusti', 'september', 'oktober', 'november', 'december')),
   ('d_t_fmt', '%a %e %b %Y %H:%M:%S'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ' '),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('sw_KE',
  [('abday', ('J2', 'J3', 'J4', 'J5', 'Alh', 'Ij', 'J1')),
   ('day', ('Jumapili', 'Jumatatu', 'Jumanne', 'Jumatano', 'Alhamisi', 'Ijumaa', 'Jumamosi')),
   ('abmon', ('Jan', 'Feb', 'Mac', 'Apr', 'Mei', 'Jun', 'Jul', 'Ago', 'Sep', 'Okt', 'Nov', 'Des')),
   ('mon', ('Januari', 'Februari', 'Machi', 'Aprili', 'Mei', 'Juni', 'Julai', 'Agosti', 'Septemba', 'Oktoba', 'Novemba', 'Desemba')),
   ('d_t_fmt', '%e %B %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%I:%M:%S %p'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[nNyY].*'),
   ('noexpr', '^[hHlL].*')]),
 ('sw_TZ',
  [('abday', ('J2', 'J3', 'J4', 'J5', 'Alh', 'Ij', 'J1')),
   ('day', ('Jumapili', 'Jumatatu', 'Jumanne', 'Jumatano', 'Alhamisi', 'Ijumaa', 'Jumamosi')),
   ('abmon', ('Jan', 'Feb', 'Mac', 'Apr', 'Mei', 'Jun', 'Jul', 'Ago', 'Sep', 'Okt', 'Nov', 'Des')),
   ('mon', ('Januari', 'Februari', 'Machi', 'Aprili', 'Mei', 'Juni', 'Julai', 'Agosti', 'Septemba', 'Oktoba', 'Novemba', 'Desemba')),
   ('d_t_fmt', '%e %B %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%I:%M:%S %p'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[nNyY].*'),
   ('noexpr', '^[hHlL].*')]),
 ('szl_PL',
  [('abday', ('niy', 'pyń', 'wtŏ', 'str', 'szt', 'pjō', 'sob')),
   ('day', ('niydziela', 'pyńdziŏek', 'wtŏrek', 'strzŏda', 'sztwortek', 'pjōntek', 'sobŏta')),
   ('abmon', ('sty', 'lut', 'mer', 'kwj', 'moj', 'czy', 'lip', 'siy', 'wrz', 'paź', 'lis', 'gru')),
   ('mon', ('styczyń', 'luty', 'merc', 'kwjeciyń', 'moj', 'czyrwjyń', 'lipjyń', 'siyrpjyń', 'wrzesiyń', 'październik', 'listopad', 'grudziyń')),
   ('d_t_fmt', '%a, %-d %b %Y, %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[JjTtYy].*'),
   ('noexpr', '^[nN].*')]),
 ('ta_IN',
  [('abday', ('ஞா', 'தி', 'செ', 'பு', 'வி', 'வெ', 'ச')),
   ('day', ('ஞாயிறு', 'திங்கள்', 'செவ்வாய்', 'புதன்', 'வியாழன்', 'வெள்ளி', 'சனி')),
   ('abmon', ('ஜன', 'பிப்', 'மார்', 'ஏப்', 'மே', 'ஜூன்', 'ஜூலை', 'ஆக', 'செப்', 'அக்', 'நவ', 'டிச')),
   ('mon', ('ஜனவரி', 'பிப்ரவரி', 'மார்ச்', 'ஏப்ரல்', 'மே', 'ஜூன்', 'ஜூலை', 'ஆகஸ்ட்', 'செப்டம்பர்', 'அக்டோபர்', 'நவம்பர்', 'டிசம்பர்')),
   ('d_t_fmt', '%A %d %B %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %B %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ஆம்yY]'),
   ('noexpr', '^[இல்லைnN]')]),
 ('ta_LK',
  [('abday', ('ஞா', 'தி', 'செ', 'பு', 'வி', 'வெ', 'ச')),
   ('day', ('ஞாயிறு', 'திங்கள்', 'செவ்வாய்', 'புதன்', 'வியாழன்', 'வெள்ளி', 'சனி')),
   ('abmon', ('ஜன', 'பிப்', 'மார்', 'ஏப்', 'மே', 'ஜூன்', 'ஜூலை', 'ஆக', 'செப்', 'அக்', 'நவ', 'டிச')),
   ('mon', ('ஜனவரி', 'பிப்ரவரி', 'மார்ச்', 'ஏப்ரல்', 'மே', 'ஜூன்', 'ஜூலை', 'ஆகஸ்ட்', 'செப்டம்பர்', 'அக்டோபர்', 'நவம்பர்', 'டிசம்பர்')),
   ('d_t_fmt', '%A %d %B %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %B %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ஆம்yY]'),
   ('noexpr', '^[இல்லைnN]')]),
 ('te_IN',
  [('abday', ('ఆది', 'సోమ', 'మంగళ', 'బుధ', 'గురు', 'శుక్ర', 'శని')),
   ('day', ('ఆదివారం', 'సోమవారం', 'మంగళవారం', 'బుధవారం', 'గురువారం', 'శుక్రవారం', 'శనివారం')),
   ('abmon', ('జనవరి', 'ఫిబ్రవరి', 'మార్చి', 'ఏప్రిల్', 'మే', 'జూన్', 'జూలై', 'ఆగస్టు', 'సెప్టెంబరు', 'అక్టోబరు', 'నవంబరు', 'డిసెంబరు')),
   ('mon', ('జనవరి', 'ఫిబ్రవరి', 'మార్చి', 'ఏప్రిల్', 'మే', 'జూన్', 'జూలై', 'ఆగస్టు', 'సెప్టెంబరు', 'అక్టోబరు', 'నవంబరు', 'డిసెంబరు')),
   ('d_t_fmt', '%B %d %A %Y %p%I.%M.%S %Z'),
   ('d_fmt', '%B %d %A %Y'),
   ('t_fmt', '%p%I.%M.%S %Z'),
   ('t_fmt_ampm', '%p%I.%M.%S %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYఅ].*'),
   ('noexpr', '^[nNక].*')]),
 ('tg_TJ',
  [('abday', ('Вск', 'Пнд', 'Втр', 'Срд', 'Чтв', 'Птн', 'Сбт')),
   ('day', ('Воскресенье', 'Понедельник', 'Вторник', 'Среда', 'Четверг', 'Пятница', 'Суббота')),
   ('abmon', ('Янв', 'Фев', 'Мар', 'Апр', 'Май', 'Июн', 'Июл', 'Авг', 'Сен', 'Окт', 'Ноя', 'Дек')),
   ('mon', ('Января', 'Февраля', 'Марта', 'Апреля', 'Мая', 'Июня', 'Июля', 'Августа', 'Сентября', 'Октября', 'Ноября', 'Декабря')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[ҲҳХхДдYy].*'),
   ('noexpr', '^[НнNn].*')]),
 ('th_TH',
  [('abday', ('อา.', 'จ.', 'อ.', 'พ.', 'พฤ.', 'ศ.', 'ส.')),
   ('day', ('อาทิตย์', 'จันทร์', 'อังคาร', 'พุธ', 'พฤหัสบดี', 'ศุกร์', 'เสาร์')),
   ('abmon', ('ม.ค.', 'ก.พ.', 'มี.ค.', 'เม.ย.', 'พ.ค.', 'มิ.ย.', 'ก.ค.', 'ส.ค.', 'ก.ย.', 'ต.ค.', 'พ.ย.', 'ธ.ค.')),
   ('mon', ('มกราคม', 'กุมภาพันธ์', 'มีนาคม', 'เมษายน', 'พฤษภาคม', 'มิถุนายน', 'กรกฎาคม', 'สิงหาคม', 'กันยายน', 'ตุลาคม', 'พฤศจิกายน', 'ธันวาคม')),
   ('d_t_fmt', '%a %e %b %Ey, %H:%M:%S'),
   ('d_fmt', '%d/%m/%Ey'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYช]'),
   ('noexpr', '^[nNม]')]),
 ('the_NP',
  [('abday', ('आइत ', 'सोम ', 'मंगल ', 'बुध ', 'बिहि ', 'शुक्र ', 'शनि ')),
   ('day', ('आइतबार ', 'सोमबार ', 'मंगलबार ', 'बुधबार ', 'बिहिबार ', 'शुक्रबार ', 'शनिबार ')),
   ('abmon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('mon', ('जनवरी', 'फ़रवरी', 'मार्च', 'अप्रेल', 'मई', 'जून', 'जुलाई', 'अगस्त', 'सितम्बर', 'अक्टूबर', 'नवम्बर', 'दिसम्बर')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('ti_ER',
  [('abday', ('ሰንበ', 'ሰኑይ', 'ሰሉስ', 'ረቡዕ', 'ሓሙስ', 'ዓርቢ', 'ቀዳም')),
   ('day', ('ሰንበት', 'ሰኑይ', 'ሰሉስ', 'ረቡዕ', 'ሓሙስ', 'ዓርቢ', 'ቀዳም')),
   ('abmon', ('ጥሪ ', 'ለካቲ', 'መጋቢ', 'ሚያዝ', 'ግንቦ', 'ሰነ ', 'ሓምለ', 'ነሓሰ', 'መስከ', 'ጥቅም', 'ሕዳር', 'ታሕሳ')),
   ('mon', ('ጥሪ', 'ለካቲት', 'መጋቢት', 'ሚያዝያ', 'ግንቦት', 'ሰነ', 'ሓምለ', 'ነሓሰ', 'መስከረም', 'ጥቅምቲ', 'ሕዳር', 'ታሕሳስ')),
   ('d_t_fmt', '%A፡ %B %e መዓልቲ %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('ti_ET',
  [('abday', ('ሰንበ', 'ሰኑይ', 'ሰሉስ', 'ረቡዕ', 'ሓሙስ', 'ዓርቢ', 'ቀዳም')),
   ('day', ('ሰንበት', 'ሰኑይ', 'ሰሉስ', 'ረቡዕ', 'ሓሙስ', 'ዓርቢ', 'ቀዳም')),
   ('abmon', ('ጃንዩ', 'ፌብሩ', 'ማርች', 'ኤፕረ', 'ሜይ ', 'ጁን ', 'ጁላይ', 'ኦገስ', 'ሴፕቴ', 'ኦክተ', 'ኖቬም', 'ዲሴም')),
   ('mon', ('ጃንዩወሪ', 'ፌብሩወሪ', 'ማርች', 'ኤፕረል', 'ሜይ', 'ጁን', 'ጁላይ', 'ኦገስት', 'ሴፕቴምበር', 'ኦክተውበር', 'ኖቬምበር', 'ዲሴምበር')),
   ('d_t_fmt', '%A፣ %B %e መዓልቲ %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('tig_ER',
  [('abday', ('ሰ/ዓ', 'ሰኖ ', 'ታላሸ', 'ኣረር', 'ከሚሽ', 'ጅምዓ', 'ሰ/ን')),
   ('day', ('ሰንበት ዓባይ', 'ሰኖ', 'ታላሸኖ', 'ኣረርባዓ', 'ከሚሽ', 'ጅምዓት', 'ሰንበት ንኢሽ')),
   ('abmon', ('ጥሪ ', 'ለካቲ', 'መጋቢ', 'ሚያዝ', 'ግንቦ', 'ሰነ ', 'ሓምለ', 'ነሓሰ', 'መስከ', 'ጥቅም', 'ሕዳር', 'ታሕሳ')),
   ('mon', ('ጥሪ', 'ለካቲት', 'መጋቢት', 'ሚያዝያ', 'ግንቦት', 'ሰነ', 'ሓምለ', 'ነሓሰ', 'መስከረም', 'ጥቅምቲ', 'ሕዳር', 'ታሕሳስ')),
   ('d_t_fmt', '%A፡ %B %e ዮም %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ''),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('tk_TM',
  [('abday', ('Duş', 'Siş', 'Çar', 'Pen', 'Ann', 'Şen', 'Ýek')),
   ('day', ('Duşenbe', 'Sişenbe', 'Çarşenbe', 'Penşenbe', 'Anna', 'Şenbe', 'Ýekşenbe')),
   ('abmon', ('Ýan', 'Few', 'Mar', 'Apr', 'Maý', 'Iýn', 'Iýl', 'Awg', 'Sen', 'Okt', 'Noý', 'Dek')),
   ('mon', ('Ýanwar', 'Fewral', 'Mart', 'Aprel', 'Maý', 'Iýun', 'Iýul', 'Awgust', 'Sentýabr', 'Oktýabr', 'Noýabr', 'Dekabr')),
   ('d_t_fmt', '%d.%m.%Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[hH].*'),
   ('noexpr', '^[ýÝnN].*')]),
 ('tl_PH',
  [('abday', ('Lin', 'Lun', 'Mar', 'Miy', 'Huw', 'Biy', 'Sab')),
   ('day', ('Linggo', 'Lunes', 'Martes', 'Miyerkoles', 'Huwebes', 'Biyernes', 'Sabado')),
   ('abmon', ('Ene', 'Peb', 'Mar', 'Abr', 'May', 'Hun', 'Hul', 'Ago', 'Sep', 'Okt', 'Nob', 'Dis')),
   ('mon', ('Enero', 'Pebrero', 'Marso', 'Abril', 'Mayo', 'Hunyo', 'Hulyo', 'Agosto', 'Septiyembre', 'Oktubre', 'Nobiyembre', 'Disyembre')),
   ('d_t_fmt', '%a %d %b %Y %r %Z'),
   ('d_fmt', '%m/%d/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('tn_ZA',
  [('abday', ('Tsh', 'Mos', 'Bed', 'Rar', 'Ne', 'Tlh', 'Mat')),
   ('day', ('laTshipi', 'Mosupologo', 'Labobedi', 'Laboraro', 'Labone', 'Labotlhano', 'Lamatlhatso')),
   ('abmon', ('Fer', 'Tlh', 'Mop', 'Mor', 'Mot', 'See', 'Phu', 'Pha', 'Lwe', 'Dip', 'Ngw', 'Sed')),
   ('mon',
    ('Ferikgong',
     'Tlhakole',
     'Mopitlwe',
     'Moranang',
     'Motsheganong',
     'Seetebosigo',
     'Phukwi',
     'Phatwe',
     'Lwetse',
     'Diphalane',
     'Ngwanatsele',
     'Sedimonthole')),
   ('d_t_fmt', '%a %-e %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYeE]'),
   ('noexpr', '^[nN]')]),
 ('tr_CY',
  [('abday', ('Paz', 'Pzt', 'Sal', 'Çrş', 'Prş', 'Cum', 'Cts')),
   ('day', ('Pazar', 'Pazartesi', 'Salı', 'Çarşamba', 'Perşembe', 'Cuma', 'Cumartesi')),
   ('abmon', ('Oca', 'Şub', 'Mar', 'Nis', 'May', 'Haz', 'Tem', 'Ağu', 'Eyl', 'Eki', 'Kas', 'Ara')),
   ('mon', ('Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[yYeE]'),
   ('noexpr', '^[nNhH]')]),
 ('tr_TR',
  [('abday', ('Paz', 'Pzt', 'Sal', 'Çrş', 'Prş', 'Cum', 'Cts')),
   ('day', ('Pazar', 'Pazartesi', 'Salı', 'Çarşamba', 'Perşembe', 'Cuma', 'Cumartesi')),
   ('abmon', ('Oca', 'Şub', 'Mar', 'Nis', 'May', 'Haz', 'Tem', 'Ağu', 'Eyl', 'Eki', 'Kas', 'Ara')),
   ('mon', ('Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d-%m-%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[yYeE]'),
   ('noexpr', '^[nNhH]')]),
 ('ts_ZA',
  [('abday', ('Son', 'Mus', 'Bir', 'Har', 'Ne', 'Tlh', 'Mug')),
   ('day', ('Sonto', 'Musumbhunuku', 'Ravumbirhi', 'Ravunharhu', 'Ravumune', 'Ravuntlhanu', 'Mugqivela')),
   ('abmon', ('Sun', 'Yan', 'Kul', 'Dzi', 'Mud', 'Kho', 'Maw', 'Mha', 'Ndz', 'Nhl', 'Huk', "N'w")),
   ('mon',
    ('Sunguti',
     'Nyenyenyani',
     'Nyenyankulu',
     'Dzivamisoko',
     'Mudyaxihi',
     'Khotavuxika',
     'Mawuwani',
     'Mhawuri',
     'Ndzhati',
     'Nhlangula',
     'Hukuri',
     "N'wendzamhala")),
   ('d_t_fmt', '%a %-e %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('tt_RU',
  [('abday', ('Якш', 'Дыш', 'Сиш', 'Чәрш', 'Пәнҗ', 'Җом', 'Шим')),
   ('day', ('Якшәмбе', 'Дышәмбе', 'Сишәмбе', 'Чәршәәмбе', 'Пәнҗешмбе', 'Җомга', 'Шимбә')),
   ('abmon', ('Янв', 'Фев', 'Мар', 'Апр', 'Май', 'Июн', 'Июл', 'Авг', 'Сен', 'Окт', 'Ноя', 'Дек')),
   ('mon', ('Января', 'Февраля', 'Марта', 'Апреля', 'Мая', 'Июня', 'Июля', 'Августа', 'Сентября', 'Октября', 'Ноября', 'Декабря')),
   ('d_t_fmt', '%a %d %b %Y %T'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^[ДдYy].*'),
   ('noexpr', '^[НнNn].*')]),
 ('ug_CN',
  [('abday', ('ي', 'د', 'س', 'چ', 'پ', 'ج', 'ش')),
   ('day', ('يەكشەنبە', 'دۈشەنبە', 'سەيشەنبە', 'چارشەنبە', 'پەيشەنبە', 'جۈمە', 'شەنبە')),
   ('abmon', ('قەھرىتان', 'ھۇت', 'نورۇز', 'ئۈمىد', 'باھار', 'سەپەر', 'چىللە', 'تومۇز', 'مىزان', 'ئوغۇز', 'ئوغلاق', 'كۆنەك')),
   ('mon', ('قەھرىتان', 'ھۇت', 'نورۇز', 'ئۈمىد', 'باھار', 'سەپەر', 'چىللە', 'تومۇز', 'مىزان', 'ئوغۇز', 'ئوغلاق', 'كۆنەك')),
   ('d_t_fmt', '%a، %d-%m-%Y، %T'),
   ('d_fmt', '%a، %d-%m-%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%T'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('uk_UA',
  [('abday', ('нд', 'пн', 'вт', 'ср', 'чт', 'пт', 'сб')),
   ('day', ('неділя', 'понеділок', 'вівторок', 'середа', 'четвер', "п'ятниця", 'субота')),
   ('abmon', ('січ', 'лют', 'бер', 'кві', 'тра', 'чер', 'лип', 'сер', 'вер', 'жов', 'лис', 'гру')),
   ('mon', ('січень', 'лютий', 'березень', 'квітень', 'травень', 'червень', 'липень', 'серпень', 'вересень', 'жовтень', 'листопад', 'грудень')),
   ('d_t_fmt', '%a, %d-%b-%Y %X %z'),
   ('d_fmt', '%d.%m.%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', '\xa0'),
   ('yesexpr', '^([Yy+]|[Тт][Аа][Кк]?)$'),
   ('noexpr', '^([Nn-]|[Нн][Іі])$')]),
 ('unm_US',
  [('abday', ('ken', 'man', 'tus', 'lel', 'tas', 'pel', 'sat')),
   ('day', ('kentuwei', 'manteke', 'tusteke', 'lelai', 'tasteke', 'pelaiteke', 'sateteke')),
   ('abmon', ('eni', 'chk', 'xam', 'kwe', 'tai', 'nip', 'lai', 'win', 'tah', 'puk', 'kun', 'mux')),
   ('mon',
    ('enikwsi', 'chkwali', 'xamokhwite', 'kwetayoxe', 'tainipen', 'kichinipen', 'lainipen', 'winaminke', 'kichitahkok', 'puksit', 'wini', 'muxkotae')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ' '),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[kKmM].*')]),
 ('ur_IN',
  [('abday', ('اتوار', 'پیر', 'منگل', 'بدھ', 'جمعرات', 'جمعہ', 'سنیچر')),
   ('day', ('اتوار', 'پیر', 'منگل', 'بدھ', 'جمعرات', 'جمعہ', 'سنیچر')),
   ('abmon', ('جنوری', 'فروری', 'مارچ', 'اپریل', 'مئی', 'جون', 'جولائی', 'اگست', 'ستمبر', 'اكتوبر', 'نومبر', 'دسمبر')),
   ('mon', ('جنوری', 'فروری', 'مارچ', 'اپریل', 'مئی', 'جون', 'جولائی', 'اگست', 'ستمبر', 'اكتوبر', 'نومبر', 'دسمبر')),
   ('d_t_fmt', '%A %d %b %Y %I:%M:%S %p %Z'),
   ('d_fmt', '%A %d %b %Y'),
   ('t_fmt', '%I:%M:%S  %Z'),
   ('t_fmt_ampm', '%I:%M:%S %p %Z'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[ہyY].*'),
   ('noexpr', '^[نnN].*')]),
 ('ur_PK',
  [('abday', ('اتوار', 'پير', 'منگل', 'بدھ', 'جمعرات', 'جمعه', 'هفته')),
   ('day', ('اتوار', 'پير', 'منگل', 'بدھ', 'جمعرات', 'جمعه', 'هفته')),
   ('abmon', ('جنوري', 'فروري', 'مارچ', 'اپريل', 'مٓی', 'جون', 'جولاي', 'اگست', 'ستمبر', 'اكتوبر', 'نومبر', 'دسمبر')),
   ('mon', ('جنوري', 'فروري', 'مارچ', 'اپريل', 'مٓی', 'جون', 'جولاي', 'اگست', 'ستمبر', 'اكتوبر', 'نومبر', 'دسمبر')),
   ('d_t_fmt', 'و %H:%M:%S %Z ت %d %B %Y'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%P %I:%M:%S'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYهبf].*'),
   ('noexpr', '^[nNنo].*')]),
 ('ve_ZA',
  [('abday', ('Swo', 'Mus', 'Vhi', 'Rar', 'ṋa', 'Ṱan', 'Mug')),
   ('day', ('Swondaha', 'Musumbuluwo', 'Ḽavhuvhili', 'Ḽavhuraru', 'Ḽavhuṋa', 'Ḽavhuṱanu', 'Mugivhela')),
   ('abmon', ('Pha', 'Luh', 'Fam', 'Lam', 'Shu', 'Lwi', 'Lwa', 'Ngu', 'Khu', 'Tsh', 'Ḽar', 'Nye')),
   ('mon',
    ('Phando', 'Luhuhi', 'Ṱhafamuhwe', 'Lambamai', 'Shundunthule', 'Fulwi', 'Fulwana', 'Ṱhangule', 'Khubvumedzi', 'Tshimedzi', 'Ḽara', 'Nyendavhusiku')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYeE]'),
   ('noexpr', '^[nNhH]')]),
 ('vi_VN',
  [('abday', ('CN', 'T2', 'T3', 'T4', 'T5', 'T6', 'T7')),
   ('day', ('Chủ nhật', 'Thứ hai', 'Thứ ba', 'Thứ tư', 'Thứ năm', 'Thứ sáu', 'Thứ bảy')),
   ('abmon', ('Th01', 'Th02', 'Th03', 'Th04', 'Th05', 'Th06', 'Th07', 'Th08', 'Th09', 'Th10', 'Th11', 'Th12')),
   ('mon',
    ('Tháng một',
     'Tháng hai',
     'Tháng ba',
     'Tháng tư',
     'Tháng năm',
     'Tháng sáu',
     'Tháng bảy',
     'Tháng tám',
     'Tháng chín',
     'Tháng mười',
     'Tháng mười một',
     'Tháng mười hai')),
   ('d_t_fmt', '%A, %d %B Năm %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', '%I:%M %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[1yYcC].*'),
   ('noexpr', '^[0nNkK].*')]),
 ('wa_BE',
  [('abday', ('dim', 'lon', 'mår', 'mie', 'dju', 'vén', 'sem')),
   ('day', ('dimegne', 'londi', 'mårdi', 'mierkidi', 'djudi', 'vénrdi', 'semdi')),
   ('abmon', ('dja', 'fev', 'mås', 'avr', 'may', 'djn', 'djl', 'awo', 'set', 'oct', 'nôv', 'dec')),
   ('mon', ('djanvî', 'fevrî', 'måss', 'avri', 'may', 'djun', 'djulete', 'awousse', 'setimbe', 'octôbe', 'nôvimbe', 'decimbe')),
   ('d_t_fmt', 'Li %A %d di %B %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', ','),
   ('thousep', '.'),
   ('yesexpr', '^[oOyYaAwW].*'),
   ('noexpr', '^[nN].*')]),
 ('wae_CH',
  [('abday', ('Sun', 'Män', 'Zis', 'Mit', 'Fro', 'Fri', 'Sam')),
   ('day', ('Suntag', 'Mäntag', 'Zischtag', 'Mittwuch', 'Frontag', 'Fritag', 'Samschtag')),
   ('abmon', ('Jen', 'Hor', 'Mär', 'Abr', 'Mei', 'Brá', 'Hei', 'Öig', 'Her', 'Wím', 'Win', 'Chr')),
   ('mon', ('Jener', 'Hornig', 'Märze', 'Abrille', 'Meije', 'Bráchet', 'Heiwet', 'Öigschte', 'Herbschtmánet', 'Wímánet', 'Wintermánet', 'Chrischtmánet')),
   ('d_t_fmt', '%a %d. %b %Y %T %Z'),
   ('d_fmt', '%Y-%m-%d'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', "'"),
   ('yesexpr', '^[jJyY].*'),
   ('noexpr', '^[nN].*')]),
 ('wal_ET',
  [('abday', ('ወጋ ', 'ሳይኖ', 'ማቆሳ', 'አሩዋ', 'ሃሙሳ', 'አርባ', 'ቄራ ')),
   ('day', ('ወጋ', 'ሳይኖ', 'ማቆሳኛ', 'አሩዋ', 'ሃሙሳ', 'አርባ', 'ቄራ')),
   ('abmon', ('ጃንዩ', 'ፌብሩ', 'ማርች', 'ኤፕረ', 'ሜይ ', 'ጁን ', 'ጁላይ', 'ኦገስ', 'ሴፕቴ', 'ኦክተ', 'ኖቬም', 'ዲሴም')),
   ('mon', ('ጃንዩወሪ', 'ፌብሩወሪ', 'ማርች', 'ኤፕረል', 'ሜይ', 'ጁን', 'ጁላይ', 'ኦገስት', 'ሴፕቴምበር', 'ኦክተውበር', 'ኖቬምበር', 'ዲሴምበር')),
   ('d_t_fmt', '%A፣ %B %e ጋላሳ %Y %r %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%l:%M:%S'),
   ('t_fmt_ampm', '%X %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY].*'),
   ('noexpr', '^[nN].*')]),
 ('wo_SN',
  [('abday', ('dib', 'alt', 'tal', 'all', 'alx', 'ajj', 'gaa')),
   ('day', ("dib'eer", 'altine', 'talaata', 'allarba', 'alxames', 'ajjuma', 'gaawu')),
   ('abmon', ('san', 'fee', 'mar', 'awr', 'me ', 'suw', 'sul', 'uut', 'sep', 'okt', 'now', 'des')),
   ('mon', ("sanwiy'e", "feebriy'e", 'mars', 'awril', 'me', 'suwen', 'sulet', 'uut', 'septaambar', 'oktoobar', 'nowaambar', 'desaambar')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d.%m.%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', ','),
   ('thousep', ''),
   ('yesexpr', '^[wWyY].*'),
   ('noexpr', '^[dDnN].*')]),
 ('xh_ZA',
  [('abday', ('Caw', 'Mvu', 'Bin', 'Tha', 'Sin', 'Hla', 'Mgq')),
   ('day', ('iCawa', 'uMvulo', 'lwesiBini', 'lwesiThathu', 'ulweSine', 'lwesiHlanu', 'uMgqibelo')),
   ('abmon', ('Mqu', 'Mdu', 'Kwi', 'Tsh', 'Can', 'Sil', 'Kha', 'Thu', 'Msi', 'Dwa', 'Nka', 'Mng')),
   ('mon',
    ('eyoMqungu',
     'eyoMdumba',
     'eyoKwindla',
     'uTshazimpuzi',
     'uCanzibe',
     'eyeSilimela',
     'eyeKhala',
     'eyeThupa',
     'eyoMsintsi',
     'eyeDwarha',
     'eyeNkanga',
     'eyoMnga')),
   ('d_t_fmt', '%a %-e %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yYeE]'),
   ('noexpr', '^[nNhH]')]),
 ('yi_US',
  [('abday', ("זונ'", "מאָנ'", "דינ'", "מיט'", "דאָנ'", "פֿרײַ'", 'שבת')),
   ('day', ('זונטיק', 'מאָנטיק', 'דינסטיק', 'מיטװאָך', 'דאָנערשטיק', 'פֿרײַטיק', 'שבת')),
   ('abmon', ('יאַנ', 'פֿעב', 'מאַר', 'אַפּר', 'מײַ ', 'יונ', 'יול', 'אױג', 'סעפּ', 'אָקט', 'נאָװ', 'דעצ')),
   ('mon', ('יאַנואַר', 'פֿעברואַר', 'מאַרץ', 'אַפּריל', 'מײַ', 'יוני', 'יולי', 'אױגוסט', 'סעפּטעמבער', 'אָקטאָבער', 'נאָװעמבער', 'דעצעמבער')),
   ('d_t_fmt', '%Z %H:%M:%S %Y %b %d %a'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%H:%M:%S'),
   ('t_fmt_ampm', '%I:%M:%S %P'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[Yyי].*'),
   ('noexpr', '^[Nnנק].*')]),
 ('yo_NG',
  [('abday', ('SUN', 'MON', 'TUE', 'WED', 'THU', 'FRI', 'SAT')),
   ('day', ('Àìkú', 'Ajé', 'Ìṣẹ́gun', 'Ọjọ́rú', 'Ọjọ́bọ̀', 'Ẹ̀tì', 'Àbámẹ́ta')),
   ('abmon', ('JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL', 'AUG', 'SEP', 'OCT', 'NOV', 'DEC')),
   ('mon', ('Jánúárì', 'Fẹ́búárì', 'Máàṣì', 'Épírì', 'Méè', 'Júùnù', 'Júláì', 'Ọ́ọ́gọsì', 'Sẹ̀tẹ̀ńbà', 'Ọtóbà', 'Nòfẹ̀ńbà', 'Dìsẹ̀ńbà')),
   ('d_t_fmt', 'ọjọ́ %A, %d oṣù %B ọdún %Y %T %Z'),
   ('d_fmt', '%d/%m/%y'),
   ('t_fmt', '%r'),
   ('t_fmt_ampm', '%I:%M:%S %p'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[EeyY].*'),
   ('noexpr', '^[rROoKkNn].*')]),
 ('yue_HK',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六')),
   ('abmon', ('1月', '2月', '3月', '4月', '5月', '6月', '7月', '8月', '9月', '10月', '11月', '12月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%Y年%m月%d日 %A %H點%M分%S秒'),
   ('d_fmt', '%Y年%m月%d日 %A'),
   ('t_fmt', '%H點%M分%S秒'),
   ('t_fmt_ampm', '%p%I點%M分%S秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('zh_CN',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六')),
   ('abmon', ('1月', '2月', '3月', '4月', '5月', '6月', '7月', '8月', '9月', '10月', '11月', '12月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%Y年%m月%d日 %A %H时%M分%S秒'),
   ('d_fmt', '%Y年%m月%d日'),
   ('t_fmt', '%H时%M分%S秒'),
   ('t_fmt_ampm', '%p %I时%M分%S秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY是]'),
   ('noexpr', '^[nN不否]')]),
 ('zh_HK',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六')),
   ('abmon', ('1月', '2月', '3月', '4月', '5月', '6月', '7月', '8月', '9月', '10月', '11月', '12月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%Y年%m月%d日 %A %H:%M:%S'),
   ('d_fmt', '%Y年%m月%d日 %A'),
   ('t_fmt', '%I時%M分%S秒 %Z'),
   ('t_fmt_ampm', '%p %I:%M:%S'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY是]'),
   ('noexpr', '^[nN不否]')]),
 ('zh_SG',
  [('abday', ('星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六')),
   ('day', ('星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六')),
   ('abmon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '%Y年%m月%d日 %H时%M分%S秒 %Z'),
   ('d_fmt', '%Y年%m月%d日'),
   ('t_fmt', '%H时%M分%S秒 %Z'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nN]')]),
 ('zh_TW',
  [('abday', ('日', '一', '二', '三', '四', '五', '六')),
   ('day', ('週日', '週一', '週二', '週三', '週四', '週五', '週六')),
   ('abmon', (' 1月', ' 2月', ' 3月', ' 4月', ' 5月', ' 6月', ' 7月', ' 8月', ' 9月', '10月', '11月', '12月')),
   ('mon', ('一月', '二月', '三月', '四月', '五月', '六月', '七月', '八月', '九月', '十月', '十一月', '十二月')),
   ('d_t_fmt', '西元%Y年%m月%d日 (%A) %H時%M分%S秒'),
   ('d_fmt', '西元%Y年%m月%d日'),
   ('t_fmt', '%H時%M分%S秒'),
   ('t_fmt_ampm', '%p %I時%M分%S秒'),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY是]'),
   ('noexpr', '^[nN不否]')]),
 ('zu_ZA',
  [('abday', ('Son', 'Mso', 'Bil', 'Tha', 'Sin', 'Hla', 'Mgq')),
   ('day', ('iSonto', 'uMsombuluko', 'uLwesibili', 'uLwesithathu', 'uLwesine', 'uLwesihlanu', 'uMgqibelo')),
   ('abmon', ('Mas', 'Ola', 'Nda', 'Mba', 'Aba', 'Ang', 'Ntu', 'Ncw', 'Man', 'Mfu', 'Lwe', 'Zib')),
   ('mon',
    ('uMasingana', 'uNhlolanja', 'uNdasa', 'uMbasa', 'uNhlaba', 'uNhlangulana', 'uNtulikazi', 'uNcwaba', 'uMandulo', 'uMfumfu', 'uLwezi', 'uZibandlela')),
   ('d_t_fmt', '%a %d %b %Y %T %Z'),
   ('d_fmt', '%d/%m/%Y'),
   ('t_fmt', '%T'),
   ('t_fmt_ampm', ''),
   ('radixchar', '.'),
   ('thousep', ','),
   ('yesexpr', '^[yY]'),
   ('noexpr', '^[nNcC]')])]
# }}}