/**
 * Copyright 2023 JogAmp Community. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of
 *       conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list
 *       of conditions and the following disclaimer in the documentation and/or other materials
 *       provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY JogAmp Community ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JogAmp Community OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are those of the
 * authors and should not be interpreted as representing official policies, either expressed
 * or implied, of JogAmp Community.
 */
package com.jogamp.gluegen;

import java.io.IOException;
import java.util.List;

/**
 * C code unit (a generated C source file),
 * covering multiple {@link FunctionEmitter} allowing to unify output, decoration and dynamic helper code injection per unit.
 **/
public class CCodeUnit extends CodeUnit {
    /** base c-unit name with suffix. */
    public final String cUnitName;

    /**
     * @param filename the class's full filename to open w/ write access
     * @param cUnitName the base c-unit name, i.e. c-file basename with suffix
     * @param generator informal optional object that is creating this unit, used to be mentioned in a warning message if not null.
     * @throws IOException
     */
    public CCodeUnit(final String filename, final String cUnitName, final Object generator) throws IOException {
        super(filename, generator);
        this.cUnitName = cUnitName;
        CodeGenUtils.emitAutogeneratedWarning(output, generator, "C-Unit: "+cUnitName+", "+filename);
    }

    public void emitHeader(final String packageName, final String className, final List<String> customCode) {
        emitln("#include <jni.h>");
        emitln("#include <stdlib.h>");
        emitln("#include <string.h>");
        emitln("#include <assert.h>");
        emitln("#include <stddef.h>");
        emitln();
        emitJNIEnvDecl();
        emitln();
        emitln("static jobject JVMUtil_NewDirectByteBufferCopy(JNIEnv *env, jclass clazzBuffers, void * source_address, size_t capacity); /* forward decl. */");
        emitln();

        boolean addNewDirectByteBufferCopyUnitCode = false;
        for (final String code : customCode) {
            emitln(code);
            addNewDirectByteBufferCopyUnitCode = addNewDirectByteBufferCopyUnitCode || code.contains("JVMUtil_NewDirectByteBufferCopy");
        }
        emitln();
        if( addNewDirectByteBufferCopyUnitCode ) {
            addTailCode(NewDirectByteBufferCopyUnitCode); // potentially used...
        }
    }

    /** Emits {@link #getJNIEnvDecl()}. */
    public void emitJNIEnvDecl() {
        emitln( getJNIEnvDecl() );
    }

    /** Emits {@link #getJNIOnLoadJNIEnvCode(String)}. */
    public void emitJNIOnLoadJNIEnvCode(final String libraryBasename) {
        emitln( getJNIOnLoadJNIEnvCode(libraryBasename) );
    }

    @Override
    public String toString() { return "CCodeUnit[unit "+cUnitName+", file "+filename+"]"; }

    public static final String NewDirectByteBufferCopyUnitCode =
        "static const char * nameCopyNativeToDirectByteBuffer = \"copyNativeToDirectByteBuffer\";\n"+
        "static const char * nameCopyNativeToDirectByteBufferSignature = \"(JJ)Ljava/nio/ByteBuffer;\";\n"+
        "\n"+
        "static jobject JVMUtil_NewDirectByteBufferCopy(JNIEnv *env, jclass clazzBuffers, void * source_address, size_t capacity) {\n"+
        "    jmethodID cstrBuffersNew = (*env)->GetStaticMethodID(env, clazzBuffers, nameCopyNativeToDirectByteBuffer, nameCopyNativeToDirectByteBufferSignature);\n"+
        "    if(NULL==cstrBuffersNew) {\n"+
        "        fprintf(stderr, \"Can't get method Buffers.%s(%s)\\n\", nameCopyNativeToDirectByteBuffer, nameCopyNativeToDirectByteBufferSignature);\n"+
        "        (*env)->FatalError(env, nameCopyNativeToDirectByteBuffer);\n"+
        "        return JNI_FALSE;\n"+
        "    }\n"+
        "    jobject jbyteBuffer  = (*env)->CallStaticObjectMethod(env, clazzBuffers, cstrBuffersNew, (jlong)(intptr_t)source_address, (jlong)capacity);\n"+
        "    if( (*env)->ExceptionCheck(env) ) {\n"+
        "        (*env)->ExceptionDescribe(env);\n"+
        "        (*env)->ExceptionClear(env);\n"+
        "        (*env)->FatalError(env, \"Exception occurred\");\n"+
        "        return NULL;\n"+
        "    }\n"+
        "    return jbyteBuffer;\n"+
        "}\n";

    /**
     * Returns native JNI declarations for `JVMUtil_GetJavaVM()`, `JVMUtil_GetJNIEnv(..)` and `JVMUtil_ReleaseJNIEnv(..)`.
     * <p>
     * See {@link #getJNIOnLoadJNIEnvCode(String)} for details.
     * </p>
     * @return the code
     * @see #getJNIOnLoadJNIEnvCode(String)
     */
    public static final String getJNIEnvDecl() {
        return "JavaVM* JVMUtil_GetJavaVM();\n"+
               "JNIEnv* JVMUtil_GetJNIEnv(int asDaemon, int* jvmAttached);\n"+
               "void JVMUtil_ReleaseJNIEnv(JNIEnv* env, int detachJVM);\n";
    }
    /**
     * Returns native JNI code `JNI_OnLoad(..)` used for dynamic libraries,
     * `JNI_OnLoad_{libraryBasename}(..)` used for static libraries,
     * `JVMUtil_GetJNIEnv(..)` etc.
     * <p>
     * The `JNI_OnLoad*(..)` methods set a `static JavaVM* {libraryBasename}_jvmHandle`,
     * which in turn is utilized by `JVMUtil_GetJNIEnv(..)`
     * to attach a new thread to the `JavaVM*` generating a new `JNIEnv*`-
     * or just to retrieve the thread's `JNIEnv*`, if already attached to the `JavaVM*`.
     * </p>
     * @param libraryBasename library basename to generate the `JNI_OnLoad_{libraryBasename}(..)` variant for statically linked libraries.
     * @return the code
     * @see #getJNIOnLoadJNIEnvDecl(String)
     */
    public static final String getJNIOnLoadJNIEnvCode(final String libraryBasename) {
        final String jvmHandleName = libraryBasename+"_jvmHandle";
        final StringBuilder sb = new StringBuilder();
        sb.append("static JavaVM *").append(jvmHandleName).append(" = NULL;\n");
        sb.append("\n");
        sb.append("JavaVM* JVMUtil_GetJavaVM() {\n");
        sb.append("  return ").append(jvmHandleName).append(";\n");
        sb.append("}\n");
        sb.append("JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *initVM, void *reserved) {\n");
        sb.append("  (void)reserved;\n");
        sb.append("  ").append(jvmHandleName).append(" = initVM;\n");
        if( GlueGen.debug() ) {
            sb.append("  fprintf(stderr, \"ON_LOAD_0 lib '").append(libraryBasename).append("'\\n\");\n");
        }
        sb.append("  return JNI_VERSION_1_6;\n");
        sb.append("}\n");
        sb.append("JNIEXPORT jint JNICALL JNI_OnLoad_").append(libraryBasename).append("(JavaVM *initVM, void *reserved) {\n");
        sb.append("  (void)reserved;\n");
        sb.append("  ").append(jvmHandleName).append(" = initVM;\n");
        if( GlueGen.debug() ) {
            sb.append("  fprintf(stderr, \"ON_LOAD_1 lib '").append(libraryBasename).append("'\\n\");\n");
        }
        sb.append("  return JNI_VERSION_1_6;\n");
        sb.append("}\n");
        sb.append("\n");
        sb.append("JNIEXPORT void JNICALL JNI_OnUnload(JavaVM *vm, void *reserved) {\n");
        sb.append("  (void)vm;\n");
        sb.append("  (void)reserved;\n");
        sb.append("  ").append(jvmHandleName).append(" = NULL;\n");
        if( GlueGen.debug() ) {
            sb.append("  fprintf(stderr, \"ON_UNLOAD_0 lib '").append(libraryBasename).append("'\\n\");\n");
        }
        sb.append("}\n");
        sb.append("\n");
        sb.append("JNIEXPORT void JNICALL JNI_OnUnload_").append(libraryBasename).append("(JavaVM *vm, void *reserved) {\n");
        sb.append("  (void)vm;\n");
        sb.append("  (void)reserved;\n");
        sb.append("  ").append(jvmHandleName).append(" = NULL;\n");
        if( GlueGen.debug() ) {
            sb.append("  fprintf(stderr, \"ON_UNLOAD_1 lib '").append(libraryBasename).append("'\\n\");\n");
        }
        sb.append("}\n");
        sb.append("\n");
        sb.append("JNIEnv* JVMUtil_GetJNIEnv(int asDaemon, int* jvmAttached) {\n");
        sb.append("    JNIEnv* curEnv = NULL;\n");
        sb.append("    JNIEnv* newEnv = NULL;\n");
        sb.append("    int envRes;\n");
        sb.append("\n");
        sb.append("    if( NULL != jvmAttached ) {\n");
        sb.append("      *jvmAttached = 0;\n");
        sb.append("    }\n");
        sb.append("    if(NULL==").append(jvmHandleName).append(") {\n");
        sb.append("        fprintf(stderr, \"JVMUtil_GetJNIEnv(").append(libraryBasename).append("): No JavaVM handle registered.\\n\");\n");
        sb.append("        return NULL;\n");
        sb.append("    }\n");
        sb.append("\n");
        sb.append("    // retrieve this thread's JNIEnv curEnv - or detect it's detached\n");
        sb.append("    envRes = (*").append(jvmHandleName).append(")->GetEnv(").append(jvmHandleName).append(", (void **) &curEnv, JNI_VERSION_1_6) ;\n");
        sb.append("    if( JNI_EDETACHED == envRes ) {\n");
        sb.append("        // detached thread - attach to JVM as daemon, w/o need to be detached!\n");
        sb.append("        if( asDaemon ) {\n");
        sb.append("            envRes = (*").append(jvmHandleName).append(")->AttachCurrentThreadAsDaemon(").append(jvmHandleName).append(", (void**) &newEnv, NULL);\n");
        sb.append("        } else {\n");
        sb.append("            envRes = (*").append(jvmHandleName).append(")->AttachCurrentThread(").append(jvmHandleName).append(", (void**) &newEnv, NULL);\n");
        sb.append("        }\n");
        sb.append("        if( JNI_OK != envRes ) {\n");
        sb.append("            fprintf(stderr, \"JVMUtil_GetJNIEnv(").append(libraryBasename).append("): Can't attach thread: %d\\n\", envRes);\n");
        sb.append("            return NULL;\n");
        sb.append("        }\n");
        sb.append("        curEnv = newEnv;\n");
        sb.append("    } else if( JNI_OK != envRes ) {\n");
        sb.append("        // oops ..\n");
        sb.append("        fprintf(stderr, \"JVMUtil_GetJNIEnv(").append(libraryBasename).append("): Can't GetEnv: %d\\n\", envRes);\n");
        sb.append("        return NULL;\n");
        sb.append("    }\n");
        sb.append("    if (curEnv==NULL) {\n");
        sb.append("        fprintf(stderr, \"JVMUtil_GetJNIEnv(").append(libraryBasename).append("): env is NULL\\n\");\n");
        sb.append("        return NULL;\n");
        sb.append("    }\n");
        sb.append("    if( NULL != jvmAttached ) {\n");
        sb.append("      *jvmAttached = NULL != newEnv;\n");
        sb.append("    }\n");
        if( GlueGen.debug() ) {
            sb.append("    fprintf(stderr, \"JVMUtil_GetJNIEnv(").append(libraryBasename).append(", asDaemon %d): jvmAttached %d -> env %p\\n.\", asDaemon, (NULL != newEnv), curEnv);\n");
        }
        sb.append("    return curEnv;\n");
        sb.append("}\n");
        sb.append("\n");
        sb.append("void JVMUtil_ReleaseJNIEnv(JNIEnv* env, int detachJVM) {\n");
        sb.append("    if(NULL==").append(jvmHandleName).append(") {\n");
        sb.append("        fprintf(stderr, \"JVMUtil_ReleaseJNIEnv(").append(libraryBasename).append("): No JavaVM handle registered.\\n\");\n");
        sb.append("        return;\n");
        sb.append("    }\n");
        sb.append("    if( detachJVM ) {\n");
        sb.append("      jint res = (*").append(jvmHandleName).append(")->DetachCurrentThread(").append(jvmHandleName).append(") ;\n");
        sb.append("      if( 0 != res ) {\n");
        sb.append("        fprintf(stderr, \"JVMUtil_ReleaseJNIEnv(").append(libraryBasename).append(", env %p): Failed with res %d\\n.\", env, res);\n");
        sb.append("        return;\n");
        sb.append("      }\n");
        if( GlueGen.debug() ) {
            sb.append("      fprintf(stderr, \"JVMUtil_ReleaseJNIEnv(").append(libraryBasename).append(", env %p) -> res %d\\n.\", env, res);\n");
        }
        sb.append("    }\n");
        sb.append("}\n");
        sb.append("\n");
        return sb.toString();
    }
}

