#! /usr/bin/env python3

import urllib.request
import json

def add_debian_packages(packages):
    import apt
    c = apt.Cache()

    for pkg in packages:
        for pkg_name in pkg['pypi_name'].lower(), pkg['name'].lower():
            if pkg_name.startswith('python-'):
                pkg_name = pkg_name[len('python-'):]
            for prefix in 'python-', 'python3-':
                debian_pkg = c.get(prefix + pkg_name)
                if debian_pkg:
                    l = pkg.setdefault('debian_pkg', [])
                    if debian_pkg.name not in l:
                        l.append(debian_pkg.name)


def add_github_stars(packages):
    import github
    import os
    g = github.Github(os.environ['GITHUB_TOKEN'])

    for pkg in packages:
        github_name = pkg['repo_url']
        if github_name.startswith('https://github.com/'):
            github_name = github_name[len('https://github.com/'):]
            repo = g.get_repo(github_name)
            pkg['repo_stars'] = len(list(repo.get_stargazers()))
        else:
            print(pkg['name'], github_name)


def add_pypi(packages):
    for pkg in packages:
        r = urllib.request.urlopen('https://pypi.python.org/pypi/'
                                   + pkg['pypi_name']
                                   + '/json')
        j = json.load(r)
        downloads = 0
        for release in j['releases'].values():
            for r in release:
                downloads += r['downloads']
        pkg['pypi_downloads'] = downloads
        pkg['pypi_license'] = j['info']['license']
        pkg['pypi_version'] = j['info']['version']

def write_package_table(fp, packages):
    for pkg in packages:
        fp.write("{debian:1} {pypi:6} {stars:4} {api:1} {prov:1} {name}\n".format(
            debian = '*' if 'debian_pkg' in pkg else ' ',
            pypi = pkg.get('pypi_downloads', 0),
            stars = pkg.get('repo_stars', 0),
            api = 'S' if pkg['stable'] else 'U',
            prov = 'P' if pkg['provisional'] else 'A',
            name = pkg['name']
        ))

    fp.write('''
Columns:
 - Debian Status:
    * package in Debian
 - Pypi downloads
 - Github Stars
 - API stability
    S stable
    U unstable
 - Accepted as Astropy Affiliated package
    A accepted
    P provisionally accepted
 - Name
''')

def update_control(packages):
    from debian import deb822
    import itertools
    cname = 'debian/control'
    with open(cname) as fp:
        header, py2, py3 = deb822.Deb822.iter_paragraphs(fp)
    
    py2_list = sorted(itertools.chain(*([p for p in pkg.get('debian_pkg', [])
                                         if p.startswith("python-")]
                                        for pkg in packages)))
    py2_old = sorted(p.strip() for p in py2['Recommends'].split(','))
    py2['Recommends'] = ',\n            '.join(py2_list)
    
    py3_list = sorted(itertools.chain(*([p for p in pkg.get('debian_pkg', [])
                                         if p.startswith("python3-")]
                                        for pkg in packages)))
    py3_old = sorted(p.strip() for p in py3['Recommends'].split(','))
    py3['Recommends'] = ',\n            '.join(py3_list)
    
    with open(cname, 'w') as fp:
        fp.write(('\n\n'.join('\n'.join(key + ": " + val
                                        for key, val in par.items())
                              for par in (header, py2, py3))))
        fp.write('\n')

    py2_added = set(py2_list) - set(py2_old)
    if len(py2_added):
        print(' * Added ' + ', '.join(py2_added)
              + ' to python-astropy-affilliated')
    py2_removed = set(py2_old) - set(py2_list)
    if len(py2_removed):
        print(' * Removed ' + ', '.join(py2_removed)
              + ' from python-astropy-affilliated')
    py3_added = set(py3_list) - set(py3_old)
    if len(py3_added):
        print(' * Added ' + ', '.join(py3_added)
              + ' to python3-astropy-affilliated')
    py3_removed = set(py3_old) - set(py3_list)
    if len(py3_removed):
        print(' * Removed ' + ', '.join(py3_removed)
              + ' from python3-astropy-affilliated')

def write_todo(fp, packages):
    for pkg in packages:
        if 'debian_pkg' in pkg:
            continue
        fp.write('''* Package name     : {name}
  Version          : {pypi_version}
  Upstream Author  : {maintainer}
* URL              : {home_url}
* License          : {pypi_license}
  Programming lang : Python
  Upstream git     : {repo_url} ({repo_stars} stars)
  Pypi URL         : https://pypi.python.org/pypi/{pypi_name} ({pypi_downloads} downloads)
  Description      : {description}

'''.format(**pkg))

r = urllib.request.urlopen("https://www.astropy.org/affiliated/registry.json")
registry = json.load(r)
packages = registry['packages']
add_debian_packages(packages)
update_control(packages)
          
add_pypi(packages)
add_github_stars(packages)
with open('affiliated.json', 'w') as fp:
    json.dump(registry, fp, indent=4)

packages.sort(key=lambda pkg: pkg['pypi_downloads'], reverse=True)
with open('affiliated-status.txt', 'w') as fp:
    write_package_table(fp, packages)

with open('TODO', 'w') as fp:
    write_todo(fp, packages)
