% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core.R
\name{nbinomWaldTest}
\alias{nbinomWaldTest}
\title{Wald test for the GLM coefficients}
\usage{
nbinomWaldTest(
  object,
  betaPrior = FALSE,
  betaPriorVar,
  modelMatrix = NULL,
  modelMatrixType,
  betaTol = 1e-08,
  maxit = 100,
  useOptim = TRUE,
  quiet = FALSE,
  useT = FALSE,
  df,
  useQR = TRUE,
  minmu = 0.5
)
}
\arguments{
\item{object}{a DESeqDataSet}

\item{betaPrior}{whether or not to put a zero-mean normal prior on
the non-intercept coefficients}

\item{betaPriorVar}{a vector with length equal to the number of
model terms including the intercept.
betaPriorVar gives the variance of the prior on the sample betas
on the log2 scale. if missing (default) this is estimated from the data}

\item{modelMatrix}{an optional matrix, typically this is set to NULL
and created within the function}

\item{modelMatrixType}{either "standard" or "expanded", which describe
how the model matrix, X of the formula in \code{\link{DESeq}}, is
formed. "standard" is as created by \code{model.matrix} using the
design formula. "expanded" includes an indicator variable for each
level of factors in addition to an intercept.
betaPrior must be set to TRUE in order for expanded model matrices
to be fit.}

\item{betaTol}{control parameter defining convergence}

\item{maxit}{the maximum number of iterations to allow for convergence of the
coefficient vector}

\item{useOptim}{whether to use the native optim function on rows which do not
converge within maxit}

\item{quiet}{whether to print messages at each step}

\item{useT}{whether to use a t-distribution as a null distribution,
for significance testing of the Wald statistics.
If FALSE, a standard normal null distribution is used.
See next argument \code{df} for information about which t is used.
If \code{useT=TRUE} then further calls to \code{\link{results}}
will make use of \code{mcols(object)$tDegreesFreedom} that is stored
by \code{nbinomWaldTest}.}

\item{df}{the degrees of freedom for the t-distribution.
This can be of length 1 or the number of rows of \code{object}.
If this is not specified, the degrees of freedom will be set
by the number of samples minus the number of columns of the design
matrix used for dispersion estimation. If \code{"weights"} are included in
the \code{assays(object)}, then the sum of the weights is used in lieu
of the number of samples.}

\item{useQR}{whether to use the QR decomposition on the design
matrix X while fitting the GLM}

\item{minmu}{lower bound on the estimated count while fitting the GLM}
}
\value{
a DESeqDataSet with results columns accessible
with the \code{\link{results}} function.  The coefficients and standard errors are
reported on a log2 scale.
}
\description{
This function tests for significance of coefficients in a Negative
Binomial GLM, using previously calculated \code{\link{sizeFactors}}
(or \code{\link{normalizationFactors}})
and dispersion estimates.  See \code{\link{DESeq}} for the GLM formula.
}
\details{
The fitting proceeds as follows: standard maximum likelihood estimates
for GLM coefficients (synonymous with "beta", "log2 fold change", "effect size")
are calculated.
Then, optionally, a zero-centered Normal prior distribution 
(\code{betaPrior}) is assumed for the coefficients other than the intercept.

Note that this posterior log2 fold change
estimation is now not the default setting for \code{nbinomWaldTest},
as the standard workflow for coefficient shrinkage has moved to
an additional function \code{link{lfcShrink}}.

For calculating Wald test p-values, the coefficients are scaled by their
standard errors and then compared to a standard Normal distribution. 
The \code{\link{results}}
function without any arguments will automatically perform a contrast of the
last level of the last variable in the design formula over the first level.
The \code{contrast} argument of the \code{\link{results}} function can be used
to generate other comparisons.
 
The Wald test can be replaced with the \code{\link{nbinomLRT}}
for an alternative test of significance.

Notes on the log2 fold change prior:

The variance of the prior distribution for each
non-intercept coefficient is calculated using the observed
distribution of the maximum likelihood coefficients.  
The final coefficients are then maximum a posteriori estimates
using this prior (Tikhonov/ridge regularization). 
See below for details on the
prior variance and the Methods section of the DESeq2 manuscript for more detail.
The use of a prior has little effect on genes with high counts and helps to
moderate the large spread in coefficients for genes with low counts.

The prior variance is calculated by matching the 0.05 upper quantile
of the observed MLE coefficients to a zero-centered Normal distribution.
In a change of methods since the 2014 paper,
the weighted upper quantile is calculated using the
\code{wtd.quantile} function from the Hmisc package
(function has been copied into DESeq2 to avoid extra dependencies).
The weights are the inverse of the expected variance of log counts, so the inverse of
\eqn{1/\bar{\mu} + \alpha_{tr}}{1/mu-bar + alpha_tr} using the mean of
normalized counts and the trended dispersion fit. The weighting ensures
that noisy estimates of log fold changes from small count genes do not
overly influence the calculation of the prior variance.
See \code{\link{estimateBetaPriorVar}}.
The final prior variance for a factor level is the average of the
estimated prior variance over all contrasts of all levels of the factor. 

When a log2 fold change prior is used (betaPrior=TRUE),
then \code{nbinomWaldTest} will by default use expanded model matrices,
as described in the \code{modelMatrixType} argument, unless this argument
is used to override the default behavior.
This ensures that log2 fold changes will be independent of the choice
of reference level. In this case, the beta prior variance for each factor
is calculated as the average of the mean squared maximum likelihood
estimates for each level and every possible contrast.
}
\examples{

dds <- makeExampleDESeqDataSet()
dds <- estimateSizeFactors(dds)
dds <- estimateDispersions(dds)
dds <- nbinomWaldTest(dds)
res <- results(dds)

}
\seealso{
\code{\link{DESeq}}, \code{\link{nbinomLRT}}
}
